
(**************************************************************************)
(*                                                                        *)
(* Module:  Unit 'PythonEngine'        Copyright (c) 1997                 *)
(*                                                                        *)
(* Version: 2.0                        Dr. Dietmar Budelsky               *)
(* Sub-Version: 0.12                   budelsky@ibs.bm.eunet.de           *)
(*                                     IBS Schillings GmbH & Co KG        *)
(*                                     Ein Unternehmen der KROHNE-Gruppe  *)
(*                                     Heisenbergstr. 18                  *)
(*                                     50169 Kerpen-Trnich               *)
(*                                     Phone: (49)22 37/97 44-0           *)
(*                                                                        *)
(*                                     Morgan Martinet                    *)
(*                                     23 rue du 14 juillet               *)
(*                                     94270 le Kremlin-Bicetre           *)
(*                                     Phone (Work): 01 47 25 70 77       *)
(*                                     e-mail: mmm@imaginet.fr            *)
(*                                                                        *)
(*  look our page at: http://www.imaginet.fr/~mmm/python.html             *)
(**************************************************************************)
(*  Functionality:  Delphi Components that provide an interface to the    *)
(*                  Python language (see python.txt for more infos on     *)
(*                  Python itself).                                       *)
(*                                                                        *)
(**************************************************************************)
(*  Contributors:                                                         *)
(*      Grzegorz Makarewicz (mak@mikroplan.com.pl)                        *)
(*      Andrew Robinson (andy@hps1.demon.co.uk)                           *)
(*      Mark Watts(mark_watts@hotmail.com)                                *)
(**************************************************************************)
(* This source code is distributed with no WARRANTY, for no reason or use.*)
(* Everyone is allowed to use and change this code free for his own tasks *)
(* and projects, as long as this header and its copyright text is intact. *)
(* For changed versions of this code, which are public distributed the    *)
(* following additional conditions have to be fullfilled:                 *)
(* 1) The header has to contain a comment on the change and the author of *)
(*    it.                                                                 *)
(* 2) A copy of the changed source has to be sent to the above E-Mail     *)
(*    address or my then valid address, if this is possible to the        *)
(*    author.                                                             *)
(* The second condition has the target to maintain an up to date central  *)
(* version of the component. If this condition is not acceptable for      *)
(* confidential or legal reasons, everyone is free to derive a component  *)
(* or to generate a diff file to my or other original sources.            *)
(* Dr. Dietmar Budelsky, 1997-11-17                                       *)
(**************************************************************************)
unit PythonEngine;

interface

uses
  Windows, Classes, SysUtils, MethodCallBack;

//#######################################################
//##                                                   ##
//##           PYTHON specific constants               ##
//##                                                   ##
//#######################################################

const
  PYTHON_DLL_NAME            = 'python15.dll';
  PYTHON_API_VERSION         = 1007;
  PYT_SCRIPT_BUFFER_INCREASE = 4096;
  PYT_METHOD_BUFFER_INCREASE = 10;

  METH_VARARGS  = 0001;
  METH_KEYWORDS = 0002;

//-------  Python opcodes  ----------//
Const
  single_input    = 256;
  file_input      = 257;
  eval_input      = 258;
  funcdef         = 259;
  parameters      = 260;
  varargslist     = 261;
  fpdef           = 262;
  fplist          = 263;
  stmt            = 264;
  simple_stmt     = 265;
  small_stmt      = 266;
  expr_stmt       = 267;
  print_stmt      = 268;
  del_stmt        = 269;
  pass_stmt       = 270;
  flow_stmt       = 271;
  break_stmt      = 272;
  continue_stmt   = 273;
  return_stmt     = 274;
  raise_stmt      = 275;
  import_stmt     = 276;
  dotted_name     = 277;
  global_stmt     = 278;
  exec_stmt       = 279;
  compound_stmt   = 280;
  if_stmt         = 281;
  while_stmt      = 282;
  for_stmt        = 283;
  try_stmt        = 284;
  except_clause   = 285;
  suite           = 286;
  test            = 287;
  and_test        = 288;
  not_test        = 289;
  comparison      = 290;
  comp_op         = 291;
  expr            = 292;
  xor_expr        = 293;
  and_expr        = 294;
  shift_expr      = 295;
  arith_expr      = 296;
  term            = 297;
  factor          = 298;
  power           = 299;
  atom            = 300;
  lambdef         = 301;
  trailer         = 302;
  subscriptlist   = 303;
  subscript       = 304;
  sliceop         = 305;
  exprlist        = 306;
  testlist        = 307;
  dictmaker       = 308;
  classdef        = 309;
  arglist         = 310;
  argument        = 311;

//#######################################################
//##                                                   ##
//##           Non-Python specific constants           ##
//##                                                   ##
//#######################################################

  ErrInit         = -300;
  CR              = #13;
  LF              = #10;
  TAB             = #09;
  CRLF            = CR+LF;



//#######################################################
//##                                                   ##
//##    Global declarations, nothing Python specific   ##
//##                                                   ##
//#######################################################

type
   TPChar  = array[0..0] of PChar;
   PPChar  = ^TPChar;
   PInt	   = ^Integer;
   PDouble = ^Double;
   PFloat  = ^Real;
   PLong   = ^LongInt;
   PShort  = ^ShortInt;
   PString = ^PChar;



//#######################################################
//##                                                   ##
//##            Python specific interface              ##
//##                                                   ##
//#######################################################

type
  PP_frozen	    = ^P_frozen;
  P_frozen	    = ^_frozen;
  PPyObject	    = ^PyObject;
  PPPyObject	    = ^PPyObject;
  PPPPyObject	    = ^PPPyObject;
  PPyIntObject	    = ^PyIntObject;
  PPyTypeObject     = ^PyTypeObject;
  PPySliceObject    = ^PySliceObject;

  AtExitProc        = procedure;
  THREADPROC        = procedure(a:Pointer);

  PyCFunction       = function( self, args:PPyObject): PPyObject; cdecl;

  unaryfunc         = function( ob1 : PPyObject): PPyObject; cdecl;
  binaryfunc        = function( ob1,ob2 : PPyObject): PPyObject; cdecl;
  ternaryfunc       = function( ob1,ob2,ob3 : PPyObject): PPyObject; cdecl;
  inquiry           = function( ob1 : PPyObject): integer; cdecl;
  coercion          = function( ob1,ob2 : PPPyObject): integer; cdecl;
  intargfunc        = function( ob1 : PPyObject; i: integer): PPyObject; cdecl;
  intintargfunc     = function( ob1 : PPyObject; i1, i2: integer):
                                PPyObject; cdecl;
  intobjargproc     = function( ob1 : PPyObject; i: integer; ob2 : PPyObject):
                                integer; cdecl;
  intintobjargproc  = function( ob1: PPyObject; i1, i2: integer;
                                ob2: PPyObject): integer; cdecl;
  objobjargproc     = function( ob1,ob2,ob3 : PPyObject): integer; cdecl;

  pydestructor      = procedure(ob: PPyObject); cdecl;
  printfunc         = function( ob: PPyObject; var f: file; i: integer): integer; cdecl;
  getattrfunc       = function( ob1: PPyObject; name: PChar): PPyObject; cdecl;
  setattrfunc       = function( ob1: PPyObject; name: PChar; ob2: PPyObject): integer; cdecl;
  cmpfunc           = function( ob1,ob2: PPyObject): integer; cdecl;
  reprfunc          = function( ob: PPyObject): PPyObject; cdecl;
  hashfunc          = function( ob: PPyObject): LongInt; cdecl;
  getattrofunc      = function( ob1,ob2: PPyObject): PPyObject; cdecl;
  setattrofunc      = function( ob1,ob2,ob3: PPyObject): integer; cdecl;

  PyNumberMethods = packed record
     nb_add	  : binaryfunc;
     nb_substract : binaryfunc;
     nb_multiply  : binaryfunc;
     nb_divide	  : binaryfunc;
     nb_remainder : binaryfunc;
     nb_divmod	  : binaryfunc;
     nb_power	  : ternaryfunc;
     nb_negative  : unaryfunc;
     nb_positive  : unaryfunc;
     nb_absolute  : unaryfunc;
     nb_nonzero	  : inquiry;
     nb_invert	  : unaryfunc;
     nb_lshift	  : binaryfunc;
     nb_rshift	  : binaryfunc;
     nb_and	  : binaryfunc;
     nb_xor	  : binaryfunc;
     nb_or	  : binaryfunc;
     nb_coerce	  : coercion;
     nb_int	  : unaryfunc;
     nb_long	  : unaryfunc;
     nb_float	  : unaryfunc;
     nb_oct	  : unaryfunc;
     nb_hex	  : unaryfunc;
  end;

  PySequenceMethods = packed record
     sq_length	  : inquiry;
     sq_concat	  : binaryfunc;
     sq_repeat	  : intargfunc;
     sq_item	  : intargfunc;
     sq_slice	  : intintargfunc;
     sq_ass_item  : intobjargproc;
     sq_ass_slice : intintobjargproc;
  end;

  PyMappingMethods = packed record
     mp_length	      : inquiry;
     mp_subscript     : binaryfunc;
     mp_ass_subscript : objobjargproc;
  end;

  Py_complex =  packed record
     real : double;
     imag : double;
  end;

  PyObject = packed record
    ob_refcnt: Integer;
    ob_type:   PPyTypeObject;
  end;

  PyIntObject = packed record
    ob_refcnt : Integer;
    ob_type   : PPyTypeObject;
    ob_ival   : LongInt;
  end;

  _frozen = packed record
     name	: PChar;
     code	: PByte;
     size	: Integer;
  end;

  PySliceObject = packed record
    ob_refcnt: Integer;
    ob_type:   PPyTypeObject;
    start, stop, step: PPyObject;
  end;

  PyTypeObject = packed record
    ob_refcnt: Integer;
    ob_type:   PPyTypeObject;
    ob_size:   Integer;
    tp_name:   PChar;
    tp_basicsize, tp_itemsize: Integer;
    tp_dealloc: pydestructor;
    tp_print:   printfunc;
    tp_getattr: getattrfunc;
    tp_setattr: setattrfunc;
    tp_compare: cmpfunc;
    tp_repr:    reprfunc;
    tp_as_number:   PyNumberMethods;
    tp_as_sequence: PySequenceMethods;
    tp_as_mapping:  PyMappingMethods;
    tp_hash:    hashfunc;
    tp_call:    ternaryfunc;
    tp_str:     reprfunc;
    tp_getattro: getattrofunc;
    tp_setattro: setattrofunc;
    tp_xxx3: LongInt;
    tp_xxx4: LongInt;
    tp_doc:  PChar;
  end;

  PPyMethodDef = ^PyMethodDef;
  PyMethodDef  = packed record
     ml_name:  PChar;
     ml_meth:  PyCFunction;
     ml_flags: Integer;
     ml_doc:   PChar;
  end;
  PPyMethodChain = ^PyMethodChain;
  PyMethodChain = packed record
    methods: PPyMethodDef;
    link:    PPyMethodChain;
  end;

  PPyClassObject = ^PyClassObject;
  PyClassObject = packed record
    // Start of the Head of an object
    ob_refcnt  : Integer;
    ob_type    : PPyTypeObject;
    // End of the Head of an object
    cl_bases   : PPyObject;       // A tuple of class objects
    cl_dict    : PPyObject;       // A dictionary
    cl_name    : PPyObject;       // A string
    // The following three are functions or NULL
    cl_getattr : PPyObject;
    cl_setattr : PPyObject;
    cl_delattr : PPyObject;
  end;

  PPyInstanceObject = ^PyInstanceObject;
  PyInstanceObject = packed record
    // Start of the Head of an object
    ob_refcnt : Integer;
    ob_type   : PPyTypeObject;
    // End of the Head of an object
    in_class  : PPyClassObject;      // The class object
    in_dict   : PPyObject;           // A dictionary
  end;

{ Instance method objects are used for two purposes:
   (a) as bound instance methods (returned by instancename.methodname)
   (b) as unbound methods (returned by ClassName.methodname)
   In case (b), im_self is NULL
}

  PPyMethodObject = ^PyMethodObject;
  PyMethodObject = packed record
    // Start of the Head of an object
    ob_refcnt : Integer;
    ob_type   : PPyTypeObject;
    // End of the Head of an object
    im_func  : PPyObject;      // The function implementing the method
    im_self  : PPyObject;      // The instance it is bound to, or NULL
    im_class : PPyObject;      // The class that defined the method
  end;


  // Bytecode object
  PPyCodeObject = ^PyCodeObject;
  PyCodeObject = packed record
    ob_refcnt      : Integer;
    ob_type        : PPyTypeObject;
    co_argcount    : Integer;         // #arguments, except *args
    co_nlocals     : Integer;         // #local variables
    co_stacksize   :Integer;          // #entries needed for evaluation stack
    co_flags       : Integer;         // CO_..., see below
    co_code        : PPyObject;       // instruction opcodes (it hides a PyStringObject)
    co_consts      : PPyObject;       // list (constants used)
    co_names       : PPyObject;       // list of strings (names used)
    co_varnames    : PPyObject;       // tuple of strings (local variable names)
    // The rest doesn't count for hash/cmp
    co_filename    : PPyObject;       // string (where it was loaded from)
    co_name        : PPyObject;       // string (name, for reference)
    co_firstlineno : Integer;         // first source line number
    co_lnotab      : PPyObject;       // string (encoding addr<->lineno mapping)
  end;


  // from frameobject.h

  PPyTryBlock = ^PyTryBlock;
  PyTryBlock = packed record
    b_type    : Integer;       // what kind of block this is
    b_handler : Integer;       // where to jump to find handler
    b_level   : Integer;       // value stack level to pop to
  end;

  PPyFrameObject = ^PyFrameObject;
  PyFrameObject = packed record
    // Start of the Head of an object
    ob_refcnt    : Integer;
    ob_type      : PPyTypeObject;
    // End of the Head of an object
    f_back       : PPyFrameObject;    // previous frame, or NULL
    f_code       : PPyCodeObject;     // code segment
    f_builtins   : PPyObject;         // builtin symbol table (PyDictObject)
    f_globals    : PPyObject;         // global symbol table (PyDictObject)
    f_locals     : PPyObject;         // local symbol table (PyDictObject)
    f_owner      : PPyObject;         // owner (e.g. class or module) or NULL
    f_fastlocals : PPyObject;         // fast local variables (PyListObject)
    f_valuestack : PPPyObject;        // malloc'ed array
    f_blockstack : PPyTryBlock;       // malloc'ed array
    f_nvalues    : Integer;           // size of f_valuestack
    f_nblocks    : Integer;           // size of f_blockstack
    f_iblock     : Integer;           // index in f_blockstack
    f_lasti      : Integer;           // Last instruction if called
    f_lineno     : Integer;           // Current line number
    f_restricted : Integer;           // Flag set if restricted operations
                                      // in this scope
    f_trace      : PPyObject;         // Trace function
  end;

  // From traceback.c
  PPyTraceBackObject = ^PyTraceBackObject;
  PyTraceBackObject = packed record
    // Start of the Head of an object
    ob_refcnt : Integer;
    ob_type   : PPyTypeObject;
    // End of the Head of an object
    tb_next   : PPyTraceBackObject;
    tb_frame  : PPyFrameObject;
    tb_lasti  : Integer;
    tb_lineno : Integer;
  end;

  // Parse tree node interface

  PNode = ^node;
  node = packed record
    n_type      : smallint;
    n_str       : PChar;
    n_lineno    : smallint;
    n_nchildren : smallint;
    n_child     : PNode;
  end;



//#######################################################
//##                                                   ##
//##         New exception classes                     ##
//##                                                   ##
//#######################################################

  // Components' exceptions
  EDLLLoadError  = class(Exception);
  EDLLImportError = class(Exception)
    public
      WrongFunc : String;
      ErrorCode : Integer;
  end;

  // Python's exceptions
  EPythonError   = class(Exception)
    public
      EName : String;
      EValue : String;
  end;
  EPyExecError   = class(EPythonError);


  // Standard exception classes of Python
  EPyException = class(EPythonError);
  EPyStandardError = class(EPyException);

  EPyAttributeError = class(EPythonError);
  EPyEOFError = class(EPyException);
  EPyIOError = class(EPyException);
  EPyImportError = class(EPyException);
  EPyKeyboardInterrupt = class(EPyException);
  EPyMemoryError = class(EPyException);
  EPyNameError = class(EPyException);
  EPyRuntimeError = class(EPyException);
  EPySystemError = class(EPyException);
  EPySystemExit = class(EPyException);
  EPyTypeError = class(EPyException);
  EPyValueError = class(EPyException);
  EPySyntaxError = class(EPyException)
    public
      EFileName   : String;
      ELineStr    : String;
      ELineNumber : Integer;
      EOffset     : Integer;
  end;

  EPyArithmeticError = class(EPyStandardError);
  EPyOverflowError = class(EPyArithmeticError);
  EPyZeroDivisionError = class(EPyArithmeticError);
  EPyFloatingPointError = class(EPyArithmeticError);

  EPyLookupError = class(EPyStandardError);
  EPyIndexError = class(EPyLookupError);
  EPyKeyError = class(EPyLookupError);




//#######################################################
//##                                                   ##
//##                   Components                      ##
//##                                                   ##
//#######################################################

//-------------------------------------------------------
//--                                                   --
//--      class:  TPythonInputOutput                   --
//--      Works as a console for Python outputs        --
//--      It's a virtual Base class                    --
//-------------------------------------------------------

const
  kMaxLines = 1000;
  kMaxLineLength = 256;

type
  TSendDataEvent = procedure (Sender: TObject; const Data : String ) of object;
  TReceiveDataEvent = procedure (Sender: TObject; var Data : String ) of object;

  TPythonInputOutput = class(TComponent)
  private
    { Dclarations prives }
  protected
    { Dclarations protges }
    FMaxLines      : Integer;
    FLine_Buffer   : String;
    FLock          : TRTLCriticalSection;
    FQueue         : TStringList;
    FDelayWrites   : Boolean;
    FMaxLineLength : Integer;
    FOnSendData    : TSendDataEvent;
    FOnReceiveData : TReceiveDataEvent;

    procedure Lock;
    procedure Unlock;
    procedure AddWrite( const str : String );
    // Virtual methods for handling the input/output of text
    procedure SendData( const Data : String ); virtual;
    function  ReceiveData : String; virtual;
    procedure AddPendingWrite; virtual;

  public
    { Dclarations publiques }
    constructor Create( AOwner : TComponent ); override;
    destructor  Destroy; override;

    procedure Write( const str : String );
    procedure WriteLine( const str : String );

  published
    { Dclarations publies }
    property MaxLines : Integer read FMaxLines write FMaxLines default kMaxLines;
    property MaxLineLength : Integer read FMaxLineLength write FMaxLineLength default kMaxLineLength;
    property DelayWrites : Boolean read FDelayWrites write FDelayWrites default False;
    property OnSendData    : TSendDataEvent read FOnSendData write FOnSendData;
    property OnReceiveData : TReceiveDataEvent read FOnReceiveData write FOnReceiveData;
  end;

//-------------------------------------------------------
//--                                                   --
//--      Base class:  TDynamicDll                     --
//--                                                   --
//-------------------------------------------------------

type
  TDynamicDll = class(TComponent)
  protected
    FDllName        : String;
    FAPIVersion     : Integer;
    FAutoLoad       : Boolean;
    FAutoUnload     : Boolean;
    FFatalMsgDlg    : Boolean;
    FFatalAbort     :  Boolean;
    FDLLHandle      : THandle;
    FOnAfterLoad    : TNotifyEvent;
    FOnBeforeUnload : TNotifyEvent;

    function  Import(const funcname: String): Pointer;
    procedure Loaded; override;
    procedure AfterLoad; virtual;
    procedure BeforeUnload; virtual;
    function  GetQuitMessage : String; virtual;

  public
    // Constructors & Destructors
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy;                    override;

    // Public methods
    procedure OpenDll(const aDllName : String);
    function  IsHandleValid : Boolean;
    procedure LoadDll;
    procedure UnloadDll;
    procedure Quit;

  published
    property AutoLoad : Boolean read FAutoLoad write FAutoLoad default True;
    property AutoUnload : Boolean read FAutoUnload write FAutoUnload default True;
    property DllName : String read FDllName write FDllName;
    property APIVersion : Integer read FAPIVersion write FAPIVersion;
    property FatalAbort :  Boolean read FFatalAbort write FFatalAbort default True;
    property FatalMsgDlg : Boolean read FFatalMsgDlg write FFatalMsgDlg default True;
    property OnAfterLoad : TNotifyEvent read FOnAfterLoad write FOnAfterLoad;
    property OnBeforeUnload : TNotifyEvent read FOnBeforeUnload write FOnBeforeUnload;
  end;

//-------------------------------------------------------
//--                                                   --
//--  class:  TPythonInterface derived from TDynamicDll--
//--      This class maps the functions imported       --
//--      from the Python Dll, and adds some           --
//--      Delphi implementations.                      --
//-------------------------------------------------------

type
  TPythonInterface=class(TDynamicDll)
  private
    DLL_PyArg_Parse: function( args: PPyObject; format: PChar {;....}):
                     Integer; cdecl;
    DLL_PyArg_ParseTuple:
                     function( args: PPyObject; format: PChar {;...}):
                     Integer; cdecl;
    DLL_Py_BuildValue:
                     function( format: PChar {;...}): PPyObject; cdecl;
    DLL_PyCode_Addr2Line:
                     function ( co: PPyCodeObject; addrq : Integer ) : Integer; cdecl;

  protected
    FInitialized:    Boolean;

    procedure AfterLoad; override;
    function  GetQuitMessage : String; override;
    procedure CheckPython;

  public
    Py_DebugFlag: PInt;
    Py_VerboseFlag: PInt;

    //_PySys_TraceFunc:    PPPyObject;
    //_PySys_ProfileFunc: PPPPyObject;

    PyImport_FrozenModules: PP_frozen;

    Py_None:            PPyObject;
    Py_Ellipsis:        PPyObject;
    Py_False:           PPyIntObject;
    Py_True:            PPyIntObject;

    PyExc_AttributeError: PPPyObject;
    //PyExc_ConflictError: PPPyObject;
    PyExc_EOFError: PPPyObject;
    PyExc_IOError: PPPyObject;
    PyExc_ImportError: PPPyObject;
    PyExc_IndexError: PPPyObject;
    PyExc_KeyError: PPPyObject;
    PyExc_KeyboardInterrupt: PPPyObject;
    PyExc_MemoryError: PPPyObject;
    PyExc_NameError: PPPyObject;
    PyExc_OverflowError: PPPyObject;
    PyExc_RuntimeError: PPPyObject;
    PyExc_SyntaxError: PPPyObject;
    PyExc_SystemError: PPPyObject;
    PyExc_SystemExit: PPPyObject;
    PyExc_TypeError: PPPyObject;
    PyExc_ValueError: PPPyObject;
    PyExc_ZeroDivisionError: PPPyObject;
    PyExc_ArithmeticError: PPPyObject;
    PyExc_Exception: PPPyObject;
    PyExc_FloatingPointError: PPPyObject;
    PyExc_LookupError: PPPyObject;
    PyExc_StandardError: PPPyObject;

    PyType_Type: PPyTypeObject;
    PyCFunction_Type: PPyTypeObject;
    PyCObject_Type: PPyTypeObject;
    PyClass_Type: PPyTypeObject;
    PyCode_Type: PPyTypeObject;
    PyComplex_Type: PPyTypeObject;
    PyDict_Type: PPyTypeObject;
    PyFile_Type: PPyTypeObject;
    PyFloat_Type: PPyTypeObject;
    PyFrame_Type: PPyTypeObject;
    PyFunction_Type: PPyTypeObject;
    PyInstance_Type: PPyTypeObject;
    PyInt_Type: PPyTypeObject;
    PyList_Type: PPyTypeObject;
    PyLong_Type: PPyTypeObject;
    PyMethod_Type: PPyTypeObject;
    PyModule_Type: PPyTypeObject;
    PyObject_Type: PPyTypeObject;
    PyRange_Type: PPyTypeObject;
    PySlice_Type: PPyTypeObject;
    PyString_Type: PPyTypeObject;
    PyTuple_Type: PPyTypeObject;

    //PyArg_GetObject: function(args : PPyObject; nargs, i: integer; p_a: PPPyObject): integer; cdecl;
    //PyArg_GetLong:   function(args : PPyObject; nargs, i: integer; p_a: PLong): integer; cdecl;
    //PyArg_GetShort:  function(args : PPyObject; nargs, i: integer; p_a: PShort): integer; cdecl;
    //PyArg_GetFloat:  function(args : PPyObject; nargs, i: integer; p_a: PFloat): integer; cdecl;
    //PyArg_GetString: function(args : PPyObject; nargs, i: integer; p_a: PString): integer; cdecl;
    //PyArgs_VaParse:  function (args : PPyObject; format: PChar; va_list: array of const): integer; cdecl;
    // Does not work!
    // Py_VaBuildValue: function (format: PChar; va_list: array of const): PPyObject; cdecl;
    //PyBuiltin_Init:     procedure; cdecl;

    PyComplex_FromCComplex: function(c: Py_complex):PPyObject; cdecl;
    PyComplex_FromDoubles: function(realv,imag : double):PPyObject; cdecl;
    PyComplex_RealAsDouble: function(op : PPyObject ): double; cdecl;
    PyComplex_ImagAsDouble: function(op : PPyObject ): double; cdecl;
    PyComplex_AsCComplex: function(op : PPyObject ): Py_complex; cdecl;
    PyCFunction_GetFunction: function(ob : PPyObject): Pointer; cdecl;
    PyCFunction_GetSelf: function(ob : PPyObject): PPyObject; cdecl;
    PyCallable_Check: function(ob	: PPyObject): integer; cdecl;
    PyCObject_FromVoidPtr: function(cobj, destruct : Pointer): PPyObject; cdecl;
    PyCObject_AsVoidPtr: function(ob : PPyObject): Pointer; cdecl;
    PyClass_New: function (ob1,ob2,ob3 :  PPyObject): PPyObject; cdecl;
    PyClass_IsSubclass: function (ob1, ob2 : PPyObject): integer cdecl;

    Py_InitModule4: function( name: PChar; methods: PPyMethodDef; doc: PChar;
                              passthrough: PPyObject; Api_Version: Integer):
                                  PPyObject; cdecl;
    PyErr_Print: procedure; cdecl;
    PyErr_SetNone: procedure(value: PPyObject); cdecl;
    PyErr_SetObject: procedure  (ob1, ob2	: PPyObject); cdecl;
    PyErr_Restore: procedure  (errtype, errvalue, errtraceback: PPyObject); cdecl;
    PyErr_BadArgument: function: integer; cdecl;
    PyErr_NoMemory: function: PPyObject; cdecl;
    PyErr_SetFromErrno: function (ob :  PPyObject):PPyObject; cdecl;
    PyErr_BadInternalCall: procedure; cdecl;
    PyErr_CheckSignals: function: integer; cdecl;
    PyErr_Occurred:     function: PPyObject; cdecl;
    PyErr_Clear:        procedure; cdecl;
    PyErr_Fetch:        procedure( errtype, errvalue, errtraceback: PPPyObject);
                                   cdecl;
    PyErr_SetString:    procedure( ErrorObject: PPyObject; text: PChar); cdecl;
    PyImport_GetModuleDict: function: PPyObject; cdecl;
    PyInt_FromLong:     function( x: LongInt):PPyObject; cdecl;
    Py_Initialize:      procedure; cdecl;
    Py_Exit:            procedure( RetVal: Integer); cdecl;
    PyEval_GetBuiltins: function: PPyObject; cdecl;
    PyDict_GetItem: function(mp, key : PPyObject):PPyObject; cdecl;
    PyDict_SetItem: function(mp, key, item :PPyObject ):integer; cdecl;
    PyDict_DelItem: function(mp, key : PPyObject ):integer; cdecl;
    PyDict_Clear: procedure(mp : PPyObject); cdecl;
    PyDict_Next: function(mp : PPyObject; pos: PInt; key, value: PPPyObject):integer; cdecl;
    PyDict_Keys: function(mp: PPyObject):PPyObject; cdecl;
    PyDict_Values: function(mp: PPyObject):PPyObject; cdecl;
    PyDict_Items: function(mp: PPyObject):PPyObject; cdecl;
    PyDict_Size: function(mp: PPyObject):integer; cdecl;
    PyDict_DelItemString: function(dp : PPyObject;key : PChar ):integer; cdecl;
    PyDict_New: function: PPyObject; cdecl;
    PyDict_GetItemString: function( dp: PPyObject; key: PChar): PPyObject; cdecl;
    PyDict_SetItemString: function( dp: PPyObject; key: PChar; item: PPyObject):
                          Integer; cdecl;
    PyModule_GetDict:     function( module:PPyObject): PPyObject; cdecl;
    PyObject_Str:         function( v: PPyObject): PPyObject; cdecl;
    PyRun_String:         function( str: PChar; start: Integer; globals: PPyObject;
                                    locals: PPyObject): PPyObject; cdecl;
    PyRun_SimpleString:   function( str: PChar): Integer; cdecl;
    PyString_AsString:    function( ob: PPyObject): PChar; cdecl;
    PyString_FromString:  function( str: PChar): PPyObject; cdecl;
    PySys_SetArgv:        procedure( argc: Integer; argv: PPChar); cdecl;

{+ means, Grzegorz has tested his non object version of this function}
{-} PyCFunction_New: function(md:PPyMethodDef;ob:PPyObject):PPyObject; cdecl;
{+} PyEval_CallObject: function(ob1,ob2:PPyObject):PPyObject; cdecl;
{-} PyEval_CallObjectWithKeywords:function (ob1,ob2,ob3:PPyObject):PPyObject; cdecl;
{-} PyEval_GetFrame:function :PPyObject; cdecl;
{-} PyEval_GetGlobals:function :PPyObject; cdecl;
{-} PyEval_GetLocals:function :PPyObject; cdecl;
{-} //PyEval_GetOwner:function :PPyObject; cdecl;
{-} PyEval_GetRestricted:function :integer; cdecl;
{-} PyEval_InitThreads:procedure; cdecl;
{-} PyEval_RestoreThread:procedure(ob:PPyObject); cdecl;
{-} PyEval_SaveThread:function :PPyObject; cdecl;
{-} PyFile_FromString:function (pc1,pc2:PChar):PPyObject; cdecl;
{-} PyFile_GetLine:function (ob:PPyObject;i:integer):PPyObject; cdecl;
{-} PyFile_Name:function (ob:PPyObject):PPyObject; cdecl;
{-} PyFile_SetBufSize:procedure(ob:PPyObject;i:integer); cdecl;
{-} PyFile_SoftSpace:function (ob:PPyObject;i:integer):integer; cdecl;
{-} PyFile_WriteObject:function (ob1,ob2:PPyObject;i:integer):integer; cdecl;
{-} PyFile_WriteString:procedure(s:PChar;ob:PPyObject); cdecl;
{+} PyFloat_AsDouble:function (ob:PPyObject):DOUBLE; cdecl;
{+} PyFloat_FromDouble:function (db:double):PPyObject; cdecl;
{-} PyFunction_GetCode:function (ob:PPyObject):PPyObject; cdecl;
{-} PyFunction_GetGlobals:function (ob:PPyObject):PPyObject; cdecl;
{-} PyFunction_New:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-} PyImport_AddModule:function (name:PChar):PPyObject; cdecl;
{-} PyImport_Cleanup:procedure; cdecl;
{-} PyImport_GetMagicNumber:function :LONGINT; cdecl;
{+} PyImport_ImportFrozenModule:function (key:PChar):integer; cdecl;
{+} PyImport_ImportModule:function (name:PChar):PPyObject; cdecl;
{-} //PyImport_Init:procedure; cdecl;
{-} PyImport_ReloadModule:function (ob:PPyObject):PPyObject; cdecl;
{-} PyInstance_New:function (obClass, obArg, obKW:PPyObject):PPyObject; cdecl;
{+} PyInt_AsLong:function (ob:PPyObject):LONGINT; cdecl;
{-} PyList_Append:function (ob1,ob2:PPyObject):integer; cdecl;
{-} PyList_AsTuple:function (ob:PPyObject):PPyObject; cdecl;
{+} PyList_GetItem:function (ob:PPyObject;i:integer):PPyObject; cdecl;
{-} PyList_GetSlice:function (ob:PPyObject;i1,i2:integer):PPyObject; cdecl;
{-} PyList_Insert:function (dp:PPyObject;idx:Integer;item:PPyObject):integer; cdecl;
{-} PyList_New:function (size:integer):PPyObject; cdecl;
{-} PyList_Reverse:function (ob:PPyObject):integer; cdecl;
{-} PyList_SetItem:function (dp:PPyObject;idx:Integer;item:PPyObject):integer; cdecl;
{-} PyList_SetSlice:function (ob:PPyObject;i1,i2:integer;ob2:PPyObject):integer; cdecl;
{+} PyList_Size:function (ob:PPyObject):integer; cdecl;
{-} PyList_Sort:function (ob:PPyObject):integer; cdecl;
{-} PyLong_AsDouble:function (ob:PPyObject):DOUBLE; cdecl;
{+} PyLong_AsLong:function (ob:PPyObject):LONGINT; cdecl;
{+} PyLong_FromDouble:function (db:double):PPyObject; cdecl;
{+} PyLong_FromLong:function (l:longint):PPyObject; cdecl;
{-} PyLong_FromString:function (pc:PChar;var ppc:PChar;i:integer):PPyObject; cdecl;
{-} PyMapping_Check:function (ob:PPyObject):integer; cdecl;
{-} PyMapping_GetItemString:function (ob:PPyObject;key:PChar):PPyObject; cdecl;
{-} PyMapping_HasKey:function (ob,key:PPyObject):integer; cdecl;
{-} PyMapping_HasKeyString:function (ob:PPyObject;key:PChar):integer; cdecl;
{-} PyMapping_Length:function (ob:PPyObject):integer; cdecl;
{-} PyMapping_SetItemString:function (ob:PPyObject; key:PChar; value:PPyObject):integer; cdecl;
{-} PyMethod_Class:function (ob:PPyObject):PPyObject; cdecl;
{-} PyMethod_Function:function (ob:PPyObject):PPyObject; cdecl;
{-} PyMethod_New:function (ob1,ob2,ob3:PPyObject):PPyObject; cdecl;
{-} PyMethod_Self:function (ob:PPyObject):PPyObject; cdecl;
{-} PyModule_GetName:function (ob:PPyObject):PChar; cdecl;
{-} PyModule_New:function (key:PChar):PPyObject; cdecl;
{-} PyNumber_Absolute:function (ob:PPyObject):PPyObject; cdecl;
{-} PyNumber_Add:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-} PyNumber_And:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-} PyNumber_Check:function (ob:PPyObject):integer; cdecl;
{-} PyNumber_Coerce:function (var ob1,ob2:PPyObject):integer; cdecl;
{-} PyNumber_Divide:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-} PyNumber_Divmod:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-} PyNumber_Float:function (ob:PPyObject):PPyObject; cdecl;
{-} PyNumber_Int:function (ob:PPyObject):PPyObject; cdecl;
{-} PyNumber_Invert:function (ob:PPyObject):PPyObject; cdecl;
{-} PyNumber_Long:function (ob:PPyObject):PPyObject; cdecl;
{-} PyNumber_Lshift:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-} PyNumber_Multiply:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-} PyNumber_Negative:function (ob:PPyObject):PPyObject; cdecl;
{-} PyNumber_Or:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-} PyNumber_Positive:function (ob:PPyObject):PPyObject; cdecl;
{-} PyNumber_Power:function (ob1,ob2,ob3:PPyObject):PPyObject; cdecl;
{-} PyNumber_Remainder:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-} PyNumber_Rshift:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-} PyNumber_Subtract:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-} PyNumber_Xor:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-} PyOS_InitInterrupts:procedure; cdecl;
{-} PyOS_InterruptOccurred:function :integer; cdecl;
{-} PyObject_CallObject:function (ob,args:PPyObject):PPyObject; cdecl;
{-} PyObject_Compare:function (ob1,ob2:PPyObject):integer; cdecl;
{-} PyObject_GetAttr:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{+} PyObject_GetAttrString:function (ob:PPyObject;c:PChar):PPyObject; cdecl;
{-} PyObject_GetItem:function (ob,key:PPyObject):PPyObject; cdecl;
{-} PyObject_HasAttrString:function (ob:PPyObject;key:PChar):integer; cdecl;
{-} PyObject_Hash:function (ob:PPyObject):LONGINT; cdecl;
{-} PyObject_IsTrue:function (ob:PPyObject):integer; cdecl;
{-} PyObject_Length:function (ob:PPyObject):integer; cdecl;
{-} PyObject_Repr:function (ob:PPyObject):PPyObject; cdecl;
{-} PyObject_SetAttr:function (ob1,ob2,ob3:PPyObject):integer; cdecl;
{-} PyObject_SetAttrString:function (ob:PPyObject;key:Pchar;value:PPyObject):integer; cdecl;
{-} PyObject_SetItem:function (ob1,ob2,ob3:PPyObject):integer; cdecl;
{-} PyRange_New:function (l1,l2,l3:longint;i:integer):PPyObject; cdecl;
{-} PySequence_Check:function (ob:PPyObject):integer; cdecl;
{-} PySequence_Concat:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-} PySequence_Count:function (ob1,ob2:PPyObject):integer; cdecl;
{-} PySequence_GetItem:function (ob:PPyObject;i:integer):PPyObject; cdecl;
{-} PySequence_GetSlice:function (ob:PPyObject;i1,i2:integer):PPyObject; cdecl;
{-} PySequence_In:function (ob1,ob2:PPyObject):integer; cdecl;
{-} PySequence_Index:function (ob1,ob2:PPyObject):integer; cdecl;
{-} PySequence_Length:function (ob:PPyObject):integer; cdecl;
{-} PySequence_Repeat:function (ob:PPyObject;count:integer):PPyObject; cdecl;
{-} PySequence_SetItem:function (ob:PPyObject;i:integer;value:PPyObject):integer; cdecl;
{-} PySequence_SetSlice:function (ob:PPyObject;i1,i2:integer;value:PPyObject):integer; cdecl;
{-} PySequence_Tuple:function (ob:PPyObject):PPyObject; cdecl;
{-} PySlice_GetIndices:function (ob:PPySliceObject;length:integer;var start,stop,step:integer):integer; cdecl;
{-} PySlice_New:function (start,stop,step:PPyObject):PPyObject; cdecl;
{-} PyString_Concat:procedure(var ob1:PPyObject;ob2:PPyObject); cdecl;
{-} PyString_ConcatAndDel:procedure(var ob1:PPyObject;ob2:PPyObject); cdecl;
{-} PyString_Format:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-} PyString_FromStringAndSize:function (s:PChar;i:integer):PPyObject; cdecl;
{-} PyString_Size:function (ob:PPyObject):integer; cdecl;
{+} PySys_GetObject:function (s:PChar):PPyObject; cdecl;
{-} //PySys_Init:procedure; cdecl;
{-} PySys_SetObject:function (s:PChar;ob:PPyObject):integer; cdecl;
{-} PySys_SetPath:procedure(path:PChar); cdecl;
{-} //PyTraceBack_Fetch:function :PPyObject; cdecl;
{-} PyTraceBack_Here:function (p:pointer):integer; cdecl;
{-} PyTraceBack_Print:function (ob1,ob2:PPyObject):integer; cdecl;
{-} //PyTraceBack_Store:function (ob:PPyObject):integer; cdecl;
{+} PyTuple_GetItem:function (ob:PPyObject;i:integer):PPyObject; cdecl;
{-} PyTuple_GetSlice:function (ob:PPyObject;i1,i2:integer):PPyObject; cdecl;
{+} PyTuple_New:function (size:Integer):PPyObject; cdecl;
{+} PyTuple_SetItem:function (ob:PPyObject;key:integer;value:PPyObject):integer; cdecl;
{+} PyTuple_Size:function (ob:PPyObject):integer; cdecl;
{-} Py_AtExit:function (proc: AtExitProc):integer; cdecl;
{-} //Py_Cleanup:procedure; cdecl;
{-} Py_CompileString:function (s1,s2:PChar;i:integer):PPyObject; cdecl;
{-} Py_FatalError:procedure(s:PChar); cdecl;
{-} Py_FindMethod:function (md:PPyMethodDef;ob:PPyObject;key:PChar):PPyObject; cdecl;
{-} Py_FindMethodInChain:function (mc:PPyMethodChain;ob:PPyObject;key:PChar):PPyObject; cdecl;
{-} Py_FlushLine:procedure; cdecl;
{-} _PyObject_New:function (obt:PPyTypeObject;ob:PPyObject):PPyObject; cdecl;
{-} _PyString_Resize:function (var ob:PPyObject;i:integer):integer; cdecl;
{-} //exit_thread:procedure; cdecl;
{-} //get_thread_ident:function :LONGINT; cdecl;
{-} //init_thread:procedure; cdecl;
{-} //start_new_thread:function (proc:THREADPROC;args:pointer):integer; cdecl;
{-} Py_Finalize                     : procedure; cdecl;
{-} PyErr_ExceptionMatches          : function ( exc : PPyObject) : Integer; cdecl;
{-} PyErr_GivenExceptionMatches     : function ( raised_exc, exc : PPyObject) : Integer; cdecl;
{-} PyEval_EvalCode                 : function ( co : PPyCodeObject; globals, locals : PPyObject) : PPyObject; cdecl;
{-} Py_GetVersion                   : function : PChar; cdecl;
{-} Py_GetCopyright                 : function : PChar; cdecl;
{-} PyParser_SimpleParseString      : function ( str : PChar; start : Integer) : PNode; cdecl;
{-} PyNode_Free                     : procedure( n : PNode ); cdecl;
{-} PyErr_NewException              : function ( name : PChar; base, dict : PPyObject ) : PPyObject; cdecl;
{-} Py_Malloc                       : function ( size : Integer ) : Pointer;
{-} PyObject_CallMethod             : function ( obj : PPyObject; method, format : PChar ) : PPyObject; cdecl;
{Further exported Objects, may be implemented later}
{
    PyCode_New: Pointer;
    PyErr_SetInterrupt: Pointer;
    PyFile_AsFile: Pointer;
    PyFile_FromFile: Pointer;
    PyFloat_AsString: Pointer;
    PyFrame_BlockPop: Pointer;
    PyFrame_BlockSetup: Pointer;
    PyFrame_ExtendStack: Pointer;
    PyFrame_FastToLocals: Pointer;
    PyFrame_LocalsToFast: Pointer;
    PyFrame_New: Pointer;
    PyGrammar_AddAccelerators: Pointer;
    PyGrammar_FindDFA: Pointer;
    PyGrammar_LabelRepr: Pointer;
    PyInstance_DoBinOp: Pointer;
    PyInt_GetMax: Pointer;
    PyMarshal_Init: Pointer;
    PyMarshal_ReadLongFromFile: Pointer;
    PyMarshal_ReadObjectFromFile: Pointer;
    PyMarshal_ReadObjectFromString: Pointer;
    PyMarshal_WriteLongToFile: Pointer;
    PyMarshal_WriteObjectToFile: Pointer;
    PyMember_Get: Pointer;
    PyMember_Set: Pointer;
    PyNode_AddChild: Pointer;
    PyNode_Compile: Pointer;
    PyNode_New: Pointer;
    PyOS_GetLastModificationTime: Pointer;
    PyOS_Readline: Pointer;
    PyOS_strtol: Pointer;
    PyOS_strtoul: Pointer;
    PyObject_CallFunction: Pointer;
    PyObject_CallMethod: Pointer;
    PyObject_Print: Pointer;
    PyParser_AddToken: Pointer;
    PyParser_Delete: Pointer;
    PyParser_New: Pointer;
    PyParser_ParseFile: Pointer;
    PyParser_ParseString: Pointer;
    PyParser_SimpleParseFile: Pointer;
    PyRun_AnyFile: Pointer;
    PyRun_File: Pointer;
    PyRun_InteractiveLoop: Pointer;
    PyRun_InteractiveOne: Pointer;
    PyRun_SimpleFile: Pointer;
    PySys_GetFile: Pointer;
    PyToken_OneChar: Pointer;
    PyToken_TwoChars: Pointer;
    PyTokenizer_Free: Pointer;
    PyTokenizer_FromFile: Pointer;
    PyTokenizer_FromString: Pointer;
    PyTokenizer_Get: Pointer;
    Py_GetExecPrefix: Pointer;
    Py_GetPath: Pointer;
    Py_GetPrefix: Pointer;
    Py_GetProgramName: Pointer;
    Py_Main: Pointer;
    _PyObject_NewVar: Pointer;
    _PyParser_Grammar: Pointer;
    _PyParser_TokenNames: Pointer;
    _PyThread_Started: Pointer;
    _Py_c_diff: Pointer;
    _Py_c_neg: Pointer;
    _Py_c_pow: Pointer;
    _Py_c_prod: Pointer;
    _Py_c_quot: Pointer;
    _Py_c_sum: Pointer;
}
  // functions redefined in Delphi
  procedure   Py_INCREF   ( op: PPyObject);
  procedure   Py_DECREF   ( op: PPyObject);
  procedure   Py_XINCREF  ( op: PPyObject);
  procedure   Py_XDECREF  ( op: PPyObject);
  function PyArg_Parse     ( args: PPyObject; format: PChar;
                             argp: array of Pointer): Integer; cdecl;
  function PyArg_ParseTuple( args: PPyObject; format: PChar;
                             argp: array of Pointer): Integer; cdecl;
// This function handles all cardinals, pointer types (with no adjustment of pointers!)
// (Extended) floats, which are handled as Python doubles and currencies, handled
// as (normalized) Python doubles.
  function Py_BuildValue( format: PChar; args: array of const): PPyObject; cdecl;
  function PyCode_Addr2Line( co: PPyCodeObject; addrq : Integer ) : Integer; cdecl;
  function PyImport_ExecCodeModule( const name : String; codeobject : PPyObject) : PPyObject;
  function PyString_Check( obj : PPyObject ) : Boolean;
  function PyFloat_Check( obj : PPyObject ) : Boolean;
  function PyInt_Check( obj : PPyObject ) : Boolean;
  function PyTuple_Check( obj : PPyObject ) : Boolean;
  function PyInstance_Check( obj : PPyObject ) : Boolean;
  function PyClass_Check( obj : PPyObject ) : Boolean;
  function PyMethod_Check( obj : PPyObject ) : Boolean;
  function PyList_Check( obj : PPyObject ) : Boolean;
  function PyDict_Check( obj : PPyObject ) : Boolean;
  function PyModule_Check( obj : PPyObject ) : Boolean;
  function PyFunction_Check( obj : PPyObject ) : Boolean;
  function Py_InitModule( const name : PChar; md : PPyMethodDef) : PPyObject;

  // Constructors & Destructors
  constructor Create(AOwner: TComponent); override;

  // Public methods
  procedure MapDll;

  // Public properties
  property Initialized : Boolean read FInitialized;
end;

//--------------------------------------------------------
//--                                                    --
//-- class:  TPythonEngine derived from TPythonInterface--
//-- Pytrunobject providing interface for               --
//-- running Python into Delphi                         --
//--------------------------------------------------------

type
  TEngineClient = class;

  TPythonEngine = class(TPythonInterface)
  private
    FInitScript:     TStrings;
    FIO:             TPythonInputOutput;
    FRedirectIO:     Boolean;
    FOnAfterInit:    TNotifyEvent;
    FClients:        TList;
    FLock:           TRTLCriticalSection;
    FExecModule:     String;
    FAutoFinalize:   Boolean;

  protected
    procedure AfterLoad; override;
    procedure Initialize;
    procedure SetInitScript(Value: TStrings);
    function  GetClientCount : Integer;
    function  GetClients( idx : Integer ) : TEngineClient;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;

  public
    // Constructors & Destructors
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;

    // Public methods
    procedure  Lock;
    procedure  Unlock;
    function   IsType(ob: PPyObject; obt: PPyTypeObject): Boolean;
    function   GetAttrString(obj: PPyObject; name: PChar):PChar;
    function   CleanString(const s : string) : string;
    function   Run_CommandAsString(const command : String; mode : Integer) : String;
    function   Run_CommandAsObject(const command : String; mode : Integer) : PPyObject;
    procedure  ExecString(const command : String);
    function   EvalString(const command : String) : PPyObject;
    function   EvalPyFunction(pyfunc, pyargs:PPyObject): Variant;
    function   EvalFunction(pyfunc:PPyObject; args: array of const): Variant;
    function   EvalFunctionNoArgs(pyfunc:PPyObject): Variant;
    function   EvalStringAsStr(const command : String) : String;
    procedure  ExecStrings( strings : TStrings );
    function   EvalStrings( strings : TStrings ) : PPyObject;
    function   EvalStringsAsStr( strings : TStrings ) : String;
    function   CheckEvalSyntax( const str : String ) : Boolean;
    function   CheckExecSyntax( const str : String ) : Boolean;
    function   CheckSyntax( const str : String; mode : Integer ) : Boolean;
    procedure  RaiseError;
    function   PyObjectAsString( obj : PPyObject ) : String;
    procedure  DoRedirectIO;
    procedure  AddClient( client : TEngineClient );
    procedure  RemoveClient( client : TEngineClient );
    function   TypeByName( const aTypeName : String ) : PPyTypeObject;
    function   ModuleByName( const aModuleName : String ) : PPyObject;
    function   MethodsByName( const aMethodsContainer: String ) : PPyMethodDef;
    function   VariantAsPyObject( V : Variant ) : PPyObject;
    function   PyObjectAsVariant( obj : PPyObject ) : Variant;
    function   VarRecAsPyObject( v : TVarRec ) : PPyObject;
    function   MakePyTuple( const objects : array of PPyObject ) : PPyObject;
    function   MakePyList( const objects : array of PPyObject ) : PPyObject;
    function   ArrayToPyTuple( items : array of const) : PPyObject;
    function   ArrayToPyList( items : array of const) : PPyObject;
    function   ArrayToPyDict( items : array of const) : PPyObject;
    function   StringsToPyList( strings : TStrings ) : PPyObject;
    function   StringsToPyTuple( strings : TStrings ) : PPyObject;
    procedure  PyListToStrings( list : PPyObject; strings : TStrings );
    procedure  PyTupleToStrings( tuple: PPyObject; strings : TStrings );
    function   ReturnNone : PPyObject;
    function   FindModule( const ModuleName : String ) : PPyObject;
    function   FindFunction(ModuleName,FuncName: String): PPyObject;

    // Public Properties
    property ClientCount : Integer read GetClientCount;
    property Clients[ idx : Integer ] : TEngineClient read GetClients;
    property ExecModule : String read FExecModule write FExecModule;

  published
    property AutoFinalize: Boolean read FAutoFinalize write FAutoFinalize default True;
    property InitScript: TStrings read FInitScript write SetInitScript;
    property OnAfterInit: TNotifyEvent read FOnAfterInit write FOnAfterInit;
    property IO: TPythonInputOutput read FIO write FIO;
    property RedirectIO: Boolean read FRedirectIO write FRedirectIO default True;
  end;


//-------------------------------------------------------
//--                                                   --
//--      Base class:  TEngineClient                   --
//--                                                   --
//-------------------------------------------------------

  TEngineClient = class(TComponent)
    protected
      FEngine : TPythonEngine;
      FOnInitialization : TNotifyEvent;
      FOnFinalization : TNotifyEvent;
      FOnCreate : TNotifyEvent;
      FOnDestroy : TNotifyEvent;
      FInitialized : Boolean;

      procedure SetEngine( val : TPythonEngine ); virtual;
      procedure Loaded; override;

    public
      // Constructors & destructors
      constructor Create( AOwner : TComponent ); override;
      destructor  Destroy; override;

      // Public Methods
      procedure Initialize; virtual;
      procedure Finalize; virtual;
      procedure ClearEngine;
      procedure CheckEngine;

      // Public Properties
      property Initialized: Boolean read FInitialized;

    published
      property Engine : TPythonEngine read FEngine write SetEngine;
      property OnCreate : TNotifyEvent read FOnCreate write FOnCreate;
      property OnDestroy : TNotifyEvent read FOnDestroy write FOnDestroy;
      property OnFinalization : TNotifyEvent read FOnFinalization write FOnFinalization;
      property OnInitialization : TNotifyEvent read FOnInitialization write FOnInitialization;
  end;

//-------------------------------------------------------
//--                                                   --
//--class: TMethodsContainer derived from TEngineClient--
//--                                                   --
//-------------------------------------------------------

  TMethodArray = array[ 0 .. 16000 ] of PyMethodDef;
  PMethodArray = ^TMethodArray;
  TDelphiMethod = function ( self, args : PPyObject ) : PPyObject of object; cdecl;

  TMethodsContainer = class(TEngineClient)
    protected
      FMethodCount : Integer;
      FAllocatedMethodCount : Integer;
      FMethods : PPyMethodDef;

      procedure AllocMethods;
      procedure FreeMethods;
      procedure ReallocMethods;
      function  GetMethods( idx : Integer ) : PPyMethodDef;

    public
      // Constructors & destructors
      constructor Create( AOwner : TComponent ); override;
      destructor  Destroy; override;

      // public methods
      procedure AddMethod( MethodName  : PChar;
                           Method  : PyCFunction;
                           DocString : PChar );
      procedure AddDelphiMethod( MethodName  : PChar;
                                 DelphiMethod: TDelphiMethod;
                                 DocString : PChar );

      // properties
      property MethodCount : Integer read FMethodCount;
      property Methods[ idx : Integer ] : PPyMethodDef read GetMethods;
      property MethodsData : PPyMethodDef read FMethods;
  end;


//-------------------------------------------------------
//--                                                   --
//--class: TPythonModule derived from TMethodsContainer--
//--                                                   --
//-------------------------------------------------------

  TPythonModule = class; // forward declaration

  TErrorType = (etString, etClass);

  TParentClassError = class(TPersistent)
    protected
      FName : String;
      FModule : String;
    public
      procedure AssignTo( Dest: TPersistent ); override;
    published
      property Module : String read FModule write FModule;
      property Name : String read FName write FName;
  end;

  TError = class(TCollectionItem)
  protected
    FName        : String;
    FText        : String;
    FError       : PPyObject;
    FErrorType   : TErrorType;
    FParentClass : TParentClassError;

    function GetDisplayName: string; override;
    procedure SetName( const Value : String );
    procedure SetText( const Value : String );
    procedure SetErrorType( Value : TErrorType );
    procedure SetParentClass( Value : TParentClassError );
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
    procedure BuildError( const ModuleName : String );
    procedure RaiseError( const msg : String );
    procedure RaiseErrorObj( const msg : String; obj : PPyObject );

    property Error : PPyObject read FError write FError;
  published
    property Name : String read FName write SetName;
    property Text : String read FText write SetText;
    property ErrorType : TErrorType read FErrorType write SetErrorType;
    property ParentClass : TParentClassError read FParentClass write SetParentClass;
  end;

  TErrors = class(TCollection)
  private
    FModule: TPythonModule;
    function GetError(Index: Integer): TError;
    procedure SetError(Index: Integer; Value: TError);
  protected
    function GetOwner: TPersistent; override;
    procedure Update(Item: TCollectionItem); override;
  public
    constructor Create(Module: TPythonModule);
    function  Add: TError;
    property Items[Index: Integer]: TError read GetError write SetError; default;
  end;

  TPythonModule = class(TMethodsContainer)
    protected
      FModuleName : String;
      FModule : PPyObject;
      FClients : TList;
      FErrors : TErrors;

      function GetClientCount : Integer;
      function GetClients( idx : Integer ) : TEngineClient;
      procedure SetErrors( val : TErrors );
      procedure SetModuleName( const val : String );

    public
      // Constructors & destructors
      constructor Create( AOwner : TComponent ); override;
      destructor  Destroy; override;

      // Public methods
      procedure MakeModule;
      procedure Initialize; override;
      procedure AddClient( client : TEngineClient );
      function  ErrorByName( const name : String ) : TError;
      procedure RaiseError( const error, msg : String );
      procedure RaiseErrorFmt( const error, format : String; Args : array of const );
      procedure RaiseErrorObj( const error, msg : String; obj : PPyObject );
      procedure BuildErrors;

      // Public properties
      property Module : PPyObject read FModule;

    published
      property ModuleName : String read FModuleName write SetModuleName;
      property ClientCount : Integer read GetClientCount;
      property Clients[ idx : Integer ] : TEngineClient read GetClients;
      property Errors : TErrors read FErrors write SetErrors;
  end;


//-------------------------------------------------------
//--                                                   --
//--class:  TPythonType  derived from TMethodsContainer--
//--                                                   --
//-------------------------------------------------------

const
  kObjectOffset = 4;

type
  TPythonType = class; //forward declaration

  // The base class of all new Python types
  TPyObject = class
    ob_refcnt      : Integer;
    ob_type        : PPyTypeObject;
    PythonType     : TPythonType;

    // Constructors & Destructors
    constructor Create( APythonType : TPythonType ); virtual;
    constructor CreateWith( APythonType : TPythonType; args : PPyObject ); virtual;

    // Misc
    function  GetSelf : PPyObject;
    procedure Adjust(PyPointer: Pointer);
    function  GetModule : TPythonModule;

    // Type services
    ////////////////

    // Basic services
    function  Print( var f: file; i: integer) : Integer; virtual;
    function  GetAttr(key : PChar) : PPyObject; virtual;
    function  SetAttr(key : PChar; value : PPyObject) : Integer; virtual;
    function  Repr : PPyObject; virtual;
    function  Compare( obj: PPyObject) : Integer; virtual;
    function  Hash : Integer; virtual;
    function  Str: PPyObject; virtual;
    function  GetAttrO( key: PPyObject) : PPyObject; virtual;
    function  SetAttrO( key, value: PPyObject) : Integer; virtual;
    function  Call( ob1, ob2 : PPyObject) : PPyObject; virtual;
    // Number services
    function  NbAdd( obj : PPyObject) : PPyObject; virtual;
    function  NbSubstract( obj : PPyObject) : PPyObject; virtual;
    function  NbMultiply( obj : PPyObject) : PPyObject; virtual;
    function  NbDivide( obj : PPyObject) : PPyObject; virtual;
    function  NbRemainder( obj : PPyObject) : PPyObject; virtual;
    function  NbDivmod( obj : PPyObject) : PPyObject; virtual;
    function  NbPower( ob1, ob2 : PPyObject) : PPyObject; virtual;
    function  NbNegative : PPyObject; virtual;
    function  NbPositive : PPyObject; virtual;
    function  NbAbsolute : PPyObject; virtual;
    function  NbNonZero : Integer; virtual;
    function  NbInvert : PPyObject; virtual;
    function  NbLShift( obj : PPyObject) : PPyObject; virtual;
    function  NbRShift( obj : PPyObject) : PPyObject; virtual;
    function  NbAnd( obj : PPyObject) : PPyObject; virtual;
    function  NbXor( obj : PPyObject) : PPyObject; virtual;
    function  NbOr( obj : PPyObject) : PPyObject; virtual;
    function  NbCoerce( obj : PPPyObject) : Integer; virtual;
    function  NbInt : PPyObject; virtual;
    function  NbLong : PPyObject; virtual;
    function  NbFloat : PPyObject; virtual;
    function  NbOct : PPyObject; virtual;
    function  NbHex : PPyObject; virtual;
    // Sequence services
    function  SqLength : Integer; virtual;
    function  SqConcat( obj : PPyObject) : PPyObject; virtual;
    function  SqRepeat( val : Integer ) : PPyObject; virtual;
    function  SqItem( idx : Integer ) : PPyObject; virtual;
    function  SqSlice( idx1, idx2 : Integer ) : PPyObject; virtual;
    function  SqAssItem( idx : integer; obj : PPyObject) : Integer; virtual;
    function  SqAssSlice( idx1, idx2 : Integer; obj : PPyObject): integer; virtual;
    // Mapping services
    function  MpLength : Integer; virtual;
    function  MpSubscript( obj : PPyObject) : PPyObject; virtual;
    function  MpAssSubscript( obj1, obj2 : PPyObject) : Integer; virtual;

    // Class methods
    class procedure RegisterMethods( PythonType : TPythonType ); virtual;
  end;
  TPyObjectClass = class of TPyObject;

  TBasicServices     = set of (bsPrint, bsGetAttr, bsSetAttr,
                               bsRepr, bsCompare, bsHash,
                               bsStr, bsGetAttrO, bsSetAttrO,
                               bsCall);
  TNumberServices    = set of (nsAdd, nsSubstract, nsMultiply,
                               nsDivide, nsRemainder, nsDivmod,
                               nsPower, nsNegative, nsPositive,
                               nsAbsolute, nsNonZero, nsInvert,
                               nsLShift, nsRShift, nsAnd,
                               nsXor, nsOr, nsCoerce,
                               nsInt, nsLong, nsFloat,
                               nsOct, nsHex);
  TSequenceServices  = set of (ssLength, ssConcat, ssRepeat,
                               ssItem, ssSlice, ssAssItem,
                               ssAssSlice);

  TMappingServices   = set of (msLength, msSubscript, msAssSubscript);

  TTypeServices = class(TPersistent)
    protected
      FBasic    : TBasicServices;
      FNumber   : TNumberServices;
      FSequence : TSequenceServices;
      FMapping  : TMappingServices;

    public
      constructor Create;
      procedure AssignTo( Dest: TPersistent ); override;

    published
      property Basic : TBasicServices read FBasic write FBasic;
      property Number : TNumberServices read FNumber write FNumber;
      property Sequence : TSequenceServices read FSequence write FSequence;
      property Mapping : TMappingServices read FMapping write FMapping;
  end;

  // The component that initializes the Python type and
  // that creates instances of itself.
  TPythonType = class(TMethodsContainer)
    protected
      FType : PyTypeObject;
      FTypeName : String;
      FModule : TPythonModule;
      FPyObjectClass : TPyObjectClass;
      FPrefix : String;
      FCreateFuncName : String;
      FServices : TTypeServices;

      procedure Notification( AComponent: TComponent;
                              Operation: TOperation); override;
      function  GetTypePtr : PPyTypeObject;
      procedure SetPyObjectClass( val : TPyObjectClass );
      procedure SetModule( val : TPythonModule );
      procedure SetServices( val : TTypeServices );
      procedure SetTypeName( const val : String );
      function  CreateMethod( pSelf, args : PPyObject ) : PPyObject; cdecl;
      procedure InitServices;

      // Type services
      // They will be all forwarded to the Delphi class that
      // implements the object through the use of virtual
      // methods
      ///////////////////////////////////////

      // Basic services

      function  Print( pSelf : PPyObject; var f: file; i: integer) : Integer; cdecl;
      function  GetAttr( pSelf : PPyObject; key : PChar ) : PPyObject; cdecl;
      function  SetAttr( pSelf : PPyObject; key : PChar; value : PPyObject) : Integer; cdecl;
      function  Repr( pSelf : PPyObject ) : PPyObject; cdecl;
      function  Compare( pSelf, obj : PPyObject ) : Integer; cdecl;
      function  Hash( pSelf : PPyObject) : Integer; cdecl;
      function  Str( pSelf : PPyObject ) : PPyObject; cdecl;
      function  GetAttrO( pSelf, key: PPyObject) : PPyObject; cdecl;
      function  SetAttrO( pSelf, key, value: PPyObject) : Integer; cdecl;
      function  Call( pSelf, ob1, ob2 : PPyObject) : PPyObject; cdecl;

      // Number services

      function  NbAdd( pSelf, obj : PPyObject) : PPyObject;
      function  NbSubstract( pSelf, obj : PPyObject) : PPyObject;
      function  NbMultiply( pSelf, obj : PPyObject) : PPyObject;
      function  NbDivide( pSelf, obj : PPyObject) : PPyObject;
      function  NbRemainder( pSelf, obj : PPyObject) : PPyObject;
      function  NbDivmod( pSelf, obj : PPyObject) : PPyObject;
      function  NbPower( pSelf, ob1, ob2 : PPyObject) : PPyObject;
      function  NbNegative( pSelf : PPyObject ) : PPyObject;
      function  NbPositive( pSelf : PPyObject ) : PPyObject;
      function  NbAbsolute( pSelf : PPyObject ) : PPyObject;
      function  NbNonZero( pSelf : PPyObject ) : Integer;
      function  NbInvert( pSelf : PPyObject ) : PPyObject;
      function  NbLShift( pSelf, obj : PPyObject) : PPyObject;
      function  NbRShift( pSelf, obj : PPyObject) : PPyObject;
      function  NbAnd( pSelf, obj : PPyObject) : PPyObject;
      function  NbXor( pSelf, obj : PPyObject) : PPyObject;
      function  NbOr( pSelf, obj : PPyObject) : PPyObject;
      function  NbCoerce( pSelf : PPyObject; obj : PPPyObject) : Integer;
      function  NbInt( pSelf : PPyObject ) : PPyObject;
      function  NbLong( pSelf : PPyObject ) : PPyObject;
      function  NbFloat( pSelf : PPyObject ) : PPyObject;
      function  NbOct( pSelf : PPyObject ) : PPyObject;
      function  NbHex( pSelf : PPyObject ) : PPyObject;

      // Sequence services

      function  SqLength( pSelf : PPyObject ) : Integer;
      function  SqConcat( pSelf, obj : PPyObject) : PPyObject;
      function  SqRepeat( pSelf : PPyObject; val : Integer ) : PPyObject;
      function  SqItem( pSelf : PPyObject; idx : Integer ) : PPyObject;
      function  SqSlice( pSelf : PPyObject; idx1, idx2 : Integer ) : PPyObject;
      function  SqAssItem( pSelf : PPyObject; idx : integer; obj : PPyObject) : Integer;
      function  SqAssSlice( pSelf : PPyObject; idx1, idx2 : Integer; obj : PPyObject): integer;

      // Mapping services

      function  MpLength( pSelf : PPyObject ) : Integer;
      function  MpSubscript( pSelf, obj : PPyObject) : PPyObject;
      function  MpAssSubscript( pSelf, obj1, obj2 : PPyObject) : Integer;

    public
      constructor Create( AOwner : TComponent ); override;
      destructor  Destroy; override;

      procedure Initialize; override;
      function  CreateInstance : PPyObject;
      function  CreateInstanceWith( args : PPyObject ) : PPyObject;
      procedure AddTypeVar;

      property TheType : PyTypeObject read FType write FType;
      property TheTypePtr : PPyTypeObject read GetTypePtr;
      property PyObjectClass : TPyObjectClass read FPyObjectClass write SetPyObjectClass stored False;

    published
      property TypeName : String read FTypeName write SetTypeName;
      property Prefix : String read FPrefix write FPrefix;
      property Module : TPythonModule read FModule write SetModule;
      property Services : TTypeServices read FServices write SetServices;
  end;

//-------------------------------------------------------
//--                                                   --
//--  class: TPythonVar derived from TEngineClient     --
//--                                                   --
//-------------------------------------------------------

  TGetDataEvent = procedure ( Sender : TObject; var Data : Variant ) of Object;
  TSetDataEvent = procedure ( Sender : TObject; Data : Variant ) of Object;

  TPythonDelphiVar = class( TEngineClient )
    protected
      FModule    : String;
      FVarName   : String;
      FVarObject : PPyObject;
      FVarType   : TPythonType;
      FOnGetData : TGetDataEvent;
      FOnSetData : TSetDataEvent;
      FOnChange  : TNotifyEvent;

      procedure CreateVarType;
      procedure CreateVar;
      function  GetValue : Variant;
      procedure SetValue( val : Variant );
      function  GetValueAsString : String;
      procedure SetVarName( const val : String );

    public
      // Constructors & Destructors
      constructor Create( AOwner : TComponent ); override;

      // Public methods
      procedure Initialize; override;
      procedure Finalize; override;

      // Public properties
      property Value : Variant read GetValue write SetValue;
      property ValueAsString : String read GetValueAsString;

    published
      property Module    : String read FModule write FModule;
      property VarName   : String read FVarName write SetVarName;
      property VarObject : PPyObject read FVarObject write FVarObject;
      property OnGetData : TGetDataEvent read FOnGetData write FOnGetData;
      property OnSetData : TSetDataEvent read FOnSetData write FOnSetData;
      property OnChange  : TNotifyEvent read FOnChange write FOnChange;
  end;


//#######################################################
//##                                                   ##
//##        New Python objects                         ##
//##                                                   ##
//#######################################################

  PPythonDelphiVar = ^PythonDelphiVar;
  PythonDelphiVar = record
    ob_refcnt      : Integer;
    ob_type        : PPyTypeObject;
    dv_var         : Variant;
    dv_component   : TPythonDelphiVar;
  end;

//#######################################################
//##                                                   ##
//##    Methods for new Python objects or modules      ##
//##                                                   ##
//#######################################################

// Module pyio for Python Input/Outputs
function  pyio_write(self, args : PPyObject) : PPyObject; cdecl;
function  pyio_read(self, args : PPyObject) : PPyObject; cdecl;
function  pyio_SetDelayWrites(self, args : PPyObject) : PPyObject; cdecl;
function  pyio_SetMaxLines(self, args : PPyObject) : PPyObject; cdecl;

// Functions for the new Type Object PythonDelphiVar
procedure PyVar_dealloc(obj : PPyObject); cdecl;
function  PyVar_getattr(obj : PPyObject; key : PChar) : PPyObject; cdecl;
function  PyVar_setattrfunc(obj : PPyObject; key : PChar; value : PPyObject) : Integer; cdecl;
function  PyVar_repr(obj : PPyObject) : PPyObject; cdecl;


//#######################################################
//##                                                   ##
//##        Global procedures                          ##
//##                                                   ##
//#######################################################

function  GetPythonEngine : TPythonEngine;
function  PythonToDelphi( obj : PPyObject ) : TPyObject;
procedure Register;


//#######################################################
//##                                                   ##
//##        Global variables                           ##
//##                                                   ##
//#######################################################


implementation


(*******************************************************)
(**                                                   **)
(**            class TPythonInputOutput               **)
(**                                                   **)
(*******************************************************)

constructor TPythonInputOutput.Create( AOwner : TComponent );
begin
  inherited;
  FMaxLines      := kMaxLines;
  FQueue         := TStringList.Create;
  FDelayWrites   := False;
  FMaxLineLength := kMaxLineLength;
  InitializeCriticalSection( FLock );
end;

destructor TPythonInputOutput.Destroy;
begin
  FQueue.Free;
  DeleteCriticalSection( FLock );
  inherited;
end;

procedure TPythonInputOutput.Lock;
begin
  EnterCriticalSection(FLock);
end;

procedure TPythonInputOutput.Unlock;
begin
  LeaveCriticalSection(FLock);
end;

procedure TPythonInputOutput.Write( const str : String );

  procedure DropLine;
  begin
    if DelayWrites then
      AddWrite( FLine_Buffer )
    else
      SendData( FLine_Buffer );
    FLine_Buffer := '';
  end;

var
  i : Integer;
  c : Char;
begin
  Lock;
  try
    for i := 1 to length(str) do
      begin
        c := str[i];
        if c = #10 then
          DropLine
        else if (c >= ' ') or (c = #09) then
          begin
            Insert( c, FLine_Buffer, length(FLine_Buffer)+1 );
            if Length(FLine_Buffer) > MaxLineLength then
              DropLine;
          end;
      end;
  finally
    Unlock;
  end;
end;

procedure TPythonInputOutput.WriteLine( const str : String );
begin
  Write( str+#10 );
end;

procedure TPythonInputOutput.AddWrite( const str : String );
begin
  FQueue.Add( str );
  if FQueue.Count > FMaxLines then
    FQueue.Delete(0)
  else
    AddPendingWrite;
end;

procedure TPythonInputOutput.SendData( const Data : String );
begin
  if Assigned(FOnSendData) then
    FOnSendData( Self, Data );
end;

function  TPythonInputOutput.ReceiveData : String;
begin
  Result := '';
  if Assigned(FOnReceiveData) then
    FOnReceiveData( Self, Result );
end;

procedure TPythonInputOutput.AddPendingWrite;
begin
end;


(*******************************************************)
(**                                                   **)
(**            class TDynamicDll                      **)
(**                                                   **)
(*******************************************************)

procedure  TDynamicDll.OpenDll(const aDllName : String);
var
  s : String;
begin
  FDLLHandle := LoadLibrary(PChar(aDllName));
  if not IsHandleValid then begin
    s := Format('Error %d: Could not open Dll "%s"',[GetLastError, aDllName]);
    if FatalMsgDlg then
      MessageBox( 0, PChar(s), 'Error', MB_TASKMODAL or MB_ICONSTOP );

    if FatalAbort then
      Quit;
  end else
    AfterLoad;
end;

constructor TDynamicDll.Create(AOwner: TComponent);
begin
  inherited;
  FFatalMsgDlg          := True;
  FFatalAbort           := True;
  FAutoLoad             := True;
end;

destructor TDynamicDll.Destroy;
begin
  if AutoUnload then
    UnloadDll;
  inherited;
end;

function TDynamicDll.Import(const funcname: String): Pointer;
var
  E : EDllImportError;
begin
  Result := GetProcAddress( FDLLHandle, PChar(funcname) );
  if (Result = nil) then begin
    E := EDllImportError.CreateFmt('Error %d: could not map symbol "%s"', [GetLastError, funcname]);
    E.ErrorCode := GetLastError;
    E.WrongFunc := funcname;
    raise E;
  end;
end;

procedure TDynamicDll.Loaded;
begin
  inherited;
  if AutoLoad and not (csDesigning in ComponentState) then
    LoadDll;
end;

function  TDynamicDll.IsHandleValid : Boolean;
begin
  Result := (FDLLHandle >= 32);
end;

procedure TDynamicDll.LoadDll;
begin
  OpenDll( DllName );
end;

procedure TDynamicDll.UnloadDll;
begin
  if IsHandleValid then begin
    BeforeUnload;
    FreeLibrary(FDLLHandle);
  end;
end;

procedure TDynamicDll.AfterLoad;
begin
  if Assigned( FOnAfterLoad ) then
    FOnAfterLoad( Self );
end;

procedure TDynamicDll.BeforeUnload;
begin
  if Assigned( FOnBeforeUnload ) then
    FOnBeforeUnload( Self );
end;

function  TDynamicDll.GetQuitMessage : String;
begin
  Result := Format( 'Dll %s could not be loaded. We must quit.', [DllName]);
end;

procedure TDynamicDll.Quit;
begin
  if not( csDesigning in ComponentState ) then begin
    MessageBox( 0, PChar(GetQuitMessage), 'Error', MB_TASKMODAL or MB_ICONSTOP );
    ExitProcess( 1 );
  end;
end;

(*******************************************************)
(**                                                   **)
(**            class TPythonInterface                 **)
(**                                                   **)
(*******************************************************)

constructor TPythonInterface.Create(AOwner: TComponent);
begin
  inherited;
  FDllName := PYTHON_DLL_NAME;
  FAPIVersion := PYTHON_API_VERSION;
  FAutoUnload := True;
end;

procedure TPythonInterface.AfterLoad;
begin
  inherited;
  try
    MapDll;
  except
    on E: Exception do begin
      if FatalMsgDlg then
        MessageBox( 0, PChar(E.Message), 'Error', MB_TASKMODAL or MB_ICONSTOP );
      if FatalAbort then Quit;
    end;
  end;
end;

function  TPythonInterface.GetQuitMessage : String;
begin
  Result := Format( 'Python could not be properly initialized. We must quit.', [DllName]);
end;

procedure TPythonInterface.CheckPython;
begin
  if not Initialized then
    raise Exception.Create('Python is not properly initialized' );
end;

procedure TPythonInterface.MapDll;
begin
  Py_DebugFlag               := Import('Py_DebugFlag');
  Py_VerboseFlag             := Import('Py_VerboseFlag');
  //_PySys_TraceFunc           := Import('_PySys_TraceFunc');
  //_PySys_ProfileFunc         := Import('_PySys_ProfileFunc');

  Py_None                    := Import('_Py_NoneStruct');
  Py_Ellipsis                := Import('_Py_EllipsisObject');
  Py_False                   := Import('_Py_ZeroStruct');
  Py_True                    := Import('_Py_TrueStruct');

  PyImport_FrozenModules     := Import('PyImport_FrozenModules');

  PyExc_AttributeError       := Import('PyExc_AttributeError');
  //PyExc_ConflictError        := Import('PyExc_ConflictError');
  PyExc_EOFError             := Import('PyExc_EOFError');
  PyExc_IOError              := Import('PyExc_IOError');
  PyExc_ImportError          := Import('PyExc_ImportError');
  PyExc_IndexError           := Import('PyExc_IndexError');
  PyExc_KeyError             := Import('PyExc_KeyError');
  PyExc_KeyboardInterrupt    := Import('PyExc_KeyboardInterrupt');
  PyExc_MemoryError          := Import('PyExc_MemoryError');
  PyExc_NameError            := Import('PyExc_NameError');
  PyExc_OverflowError        := Import('PyExc_OverflowError');
  PyExc_RuntimeError         := Import('PyExc_RuntimeError');
  PyExc_SyntaxError          := Import('PyExc_SyntaxError');
  PyExc_SystemError          := Import('PyExc_SystemError');
  PyExc_SystemExit           := Import('PyExc_SystemExit');
  PyExc_TypeError            := Import('PyExc_TypeError');
  PyExc_ValueError           := Import('PyExc_ValueError');
  PyExc_ZeroDivisionError    := Import('PyExc_ZeroDivisionError');
  PyExc_ArithmeticError      := Import('PyExc_ArithmeticError');
  PyExc_Exception            := Import('PyExc_Exception');
  PyExc_FloatingPointError   := Import('PyExc_FloatingPointError');
  PyExc_LookupError          := Import('PyExc_LookupError');
  PyExc_StandardError        := Import('PyExc_StandardError');

  PyType_Type                := Import('PyType_Type');
  PyCFunction_Type           := Import('PyCFunction_Type');
  PyCObject_Type             := Import('PyCObject_Type');
  PyClass_Type               := Import('PyClass_Type');
  PyCode_Type                := Import('PyCode_Type');
  PyComplex_Type             := Import('PyComplex_Type');
  PyDict_Type                := Import('PyDict_Type');
  PyFile_Type                := Import('PyFile_Type');
  PyFloat_Type               := Import('PyFloat_Type');
  PyFrame_Type               := Import('PyFrame_Type');
  PyFunction_Type            := Import('PyFunction_Type');
  PyInstance_Type            := Import('PyInstance_Type');
  PyInt_Type                 := Import('PyInt_Type');
  PyList_Type                := Import('PyList_Type');
  PyLong_Type                := Import('PyLong_Type');
  PyMethod_Type              := Import('PyMethod_Type');
  PyModule_Type              := Import('PyModule_Type');
  PyObject_Type              := Import('PyObject_Type');
  PyRange_Type               := Import('PyRange_Type');
  PySlice_Type               := Import('PySlice_Type');
  PyString_Type              := Import('PyString_Type');
  PyTuple_Type               := Import('PyTuple_Type');

  //@PyArg_GetObject           := Import('PyArg_GetObject');
  //@PyArg_GetLong             := Import('PyArg_GetLong');
  //@PyArg_GetShort            := Import('PyArg_GetShort');
  //@PyArg_GetFloat            := Import('PyArg_GetFloat');
  //@PyArg_GetString           := Import('PyArg_GetString');
  //@PyArgs_VaParse            := Import('PyArgs_VaParse');
  //@Py_VaBuildValue           := Import('Py_VaBuildValue');
  //@PyBuiltin_Init            := Import('PyBuiltin_Init');
  @PyComplex_FromCComplex    := Import('PyComplex_FromCComplex');
  @PyComplex_FromDoubles     := Import('PyComplex_FromDoubles');
  @PyComplex_RealAsDouble    := Import('PyComplex_RealAsDouble');
  @PyComplex_ImagAsDouble    := Import('PyComplex_ImagAsDouble');
  @PyComplex_AsCComplex      := Import('PyComplex_AsCComplex');
  @PyCFunction_GetFunction   := Import('PyCFunction_GetFunction');
  @PyCFunction_GetSelf       := Import('PyCFunction_GetSelf');
  @PyCallable_Check          := Import('PyCallable_Check');
  @PyCObject_FromVoidPtr     := Import('PyCObject_FromVoidPtr');
  @PyCObject_AsVoidPtr       := Import('PyCObject_AsVoidPtr');
  @PyClass_New               := Import('PyClass_New');
  @PyClass_IsSubclass        := Import('PyClass_IsSubclass');
  @PyDict_GetItem            := Import('PyDict_GetItem');
  @PyDict_SetItem            := Import('PyDict_SetItem');
  @PyDict_DelItem            := Import('PyDict_DelItem');
  @PyDict_Clear              := Import('PyDict_Clear');
  @PyDict_Next               := Import('PyDict_Next');
  @PyDict_Keys               := Import('PyDict_Keys');
  @PyDict_Values             := Import('PyDict_Values');
  @PyDict_Items              := Import('PyDict_Items');
  @PyDict_Size               := Import('PyDict_Size');
  @PyDict_DelItemString      := Import('PyDict_DelItemString');
  @Py_InitModule4            := Import('Py_InitModule4');
  @PyErr_Print               := Import('PyErr_Print');
  @PyErr_SetNone             := Import('PyErr_SetNone');
  @PyErr_SetObject           := Import('PyErr_SetObject');
  @PyErr_Restore             := Import('PyErr_Restore');
  @PyErr_BadArgument         := Import('PyErr_BadArgument');
  @PyErr_NoMemory            := Import('PyErr_NoMemory');
  @PyErr_SetFromErrno        := Import('PyErr_SetFromErrno');
  @PyErr_BadInternalCall     := Import('PyErr_BadInternalCall');
  @PyErr_CheckSignals        := Import('PyErr_CheckSignals');
  @PyErr_Occurred            := Import('PyErr_Occurred');
  @PyErr_Clear               := Import('PyErr_Clear');
  @PyErr_Fetch               := Import('PyErr_Fetch');
  @PyErr_SetString           := Import('PyErr_SetString');
  @PyEval_GetBuiltins        := Import('PyEval_GetBuiltins');
  @PyImport_GetModuleDict    := Import('PyImport_GetModuleDict');
  @PyInt_FromLong            := Import('PyInt_FromLong');
  @DLL_PyArg_ParseTuple      := Import('PyArg_ParseTuple');
  @DLL_PyArg_Parse           := Import('PyArg_Parse');
  @DLL_Py_BuildValue         := Import('Py_BuildValue');
  @Py_Initialize             := Import('Py_Initialize');
  @PyDict_New                := Import('PyDict_New');
  @PyDict_SetItemString      := Import('PyDict_SetItemString');
  @PyModule_GetDict          := Import('PyModule_GetDict');
  @PyObject_Str              := Import('PyObject_Str');
  @PyRun_String              := Import('PyRun_String');
  @PyRun_SimpleString        := Import('PyRun_SimpleString');
  @PyDict_GetItemString      := Import('PyDict_GetItemString');
  @PyString_AsString         := Import('PyString_AsString');
  @PyString_FromString       := Import('PyString_FromString');
  @PySys_SetArgv             := Import('PySys_SetArgv');
  @Py_Exit                   := Import('Py_Exit');

  @PyCFunction_New           :=Import('PyCFunction_New');
  @PyEval_CallObject         :=Import('PyEval_CallObject');
  @PyEval_CallObjectWithKeywords:=Import('PyEval_CallObjectWithKeywords');
  @PyEval_GetFrame           :=Import('PyEval_GetFrame');
  @PyEval_GetGlobals         :=Import('PyEval_GetGlobals');
  @PyEval_GetLocals          :=Import('PyEval_GetLocals');
  //@PyEval_GetOwner           :=Import('PyEval_GetOwner');
  @PyEval_GetRestricted      :=Import('PyEval_GetRestricted');
  @PyEval_InitThreads        :=Import('PyEval_InitThreads');
  @PyEval_RestoreThread      :=Import('PyEval_RestoreThread');
  @PyEval_SaveThread         :=Import('PyEval_SaveThread');
  @PyFile_FromString         :=Import('PyFile_FromString');
  @PyFile_GetLine            :=Import('PyFile_GetLine');
  @PyFile_Name               :=Import('PyFile_Name');
  @PyFile_SetBufSize         :=Import('PyFile_SetBufSize');
  @PyFile_SoftSpace          :=Import('PyFile_SoftSpace');
  @PyFile_WriteObject        :=Import('PyFile_WriteObject');
  @PyFile_WriteString        :=Import('PyFile_WriteString');
  @PyFloat_AsDouble          :=Import('PyFloat_AsDouble');
  @PyFloat_FromDouble        :=Import('PyFloat_FromDouble');
  @PyFunction_GetCode        :=Import('PyFunction_GetCode');
  @PyFunction_GetGlobals     :=Import('PyFunction_GetGlobals');
  @PyFunction_New            :=Import('PyFunction_New');
  @PyImport_AddModule        :=Import('PyImport_AddModule');
  @PyImport_Cleanup          :=Import('PyImport_Cleanup');
  @PyImport_GetMagicNumber   :=Import('PyImport_GetMagicNumber');
  @PyImport_ImportFrozenModule:=Import('PyImport_ImportFrozenModule');
  @PyImport_ImportModule     :=Import('PyImport_ImportModule');
  //@PyImport_Init             :=Import('PyImport_Init');
  @PyImport_ReloadModule     :=Import('PyImport_ReloadModule');
  @PyInstance_New            :=Import('PyInstance_New');
  @PyInt_AsLong              :=Import('PyInt_AsLong');
  @PyList_Append             :=Import('PyList_Append');
  @PyList_AsTuple            :=Import('PyList_AsTuple');
  @PyList_GetItem            :=Import('PyList_GetItem');
  @PyList_GetSlice           :=Import('PyList_GetSlice');
  @PyList_Insert             :=Import('PyList_Insert');
  @PyList_New                :=Import('PyList_New');
  @PyList_Reverse            :=Import('PyList_Reverse');
  @PyList_SetItem            :=Import('PyList_SetItem');
  @PyList_SetSlice           :=Import('PyList_SetSlice');
  @PyList_Size               :=Import('PyList_Size');
  @PyList_Sort               :=Import('PyList_Sort');
  @PyLong_AsDouble           :=Import('PyLong_AsDouble');
  @PyLong_AsLong             :=Import('PyLong_AsLong');
  @PyLong_FromDouble         :=Import('PyLong_FromDouble');
  @PyLong_FromLong           :=Import('PyLong_FromLong');
  @PyLong_FromString         :=Import('PyLong_FromString');
  @PyMapping_Check           :=Import('PyMapping_Check');
  @PyMapping_GetItemString   :=Import('PyMapping_GetItemString');
  @PyMapping_HasKey          :=Import('PyMapping_HasKey');
  @PyMapping_HasKeyString    :=Import('PyMapping_HasKeyString');
  @PyMapping_Length          :=Import('PyMapping_Length');
  @PyMapping_SetItemString   :=Import('PyMapping_SetItemString');
  @PyMethod_Class            :=Import('PyMethod_Class');
  @PyMethod_Function         :=Import('PyMethod_Function');
  @PyMethod_New              :=Import('PyMethod_New');
  @PyMethod_Self             :=Import('PyMethod_Self');
  @PyModule_GetName          :=Import('PyModule_GetName');
  @PyModule_New              :=Import('PyModule_New');
  @PyNumber_Absolute         :=Import('PyNumber_Absolute');
  @PyNumber_Add              :=Import('PyNumber_Add');
  @PyNumber_And              :=Import('PyNumber_And');
  @PyNumber_Check            :=Import('PyNumber_Check');
  @PyNumber_Coerce           :=Import('PyNumber_Coerce');
  @PyNumber_Divide           :=Import('PyNumber_Divide');
  @PyNumber_Divmod           :=Import('PyNumber_Divmod');
  @PyNumber_Float            :=Import('PyNumber_Float');
  @PyNumber_Int              :=Import('PyNumber_Int');
  @PyNumber_Invert           :=Import('PyNumber_Invert');
  @PyNumber_Long             :=Import('PyNumber_Long');
  @PyNumber_Lshift           :=Import('PyNumber_Lshift');
  @PyNumber_Multiply         :=Import('PyNumber_Multiply');
  @PyNumber_Negative         :=Import('PyNumber_Negative');
  @PyNumber_Or               :=Import('PyNumber_Or');
  @PyNumber_Positive         :=Import('PyNumber_Positive');
  @PyNumber_Power            :=Import('PyNumber_Power');
  @PyNumber_Remainder        :=Import('PyNumber_Remainder');
  @PyNumber_Rshift           :=Import('PyNumber_Rshift');
  @PyNumber_Subtract         :=Import('PyNumber_Subtract');
  @PyNumber_Xor              :=Import('PyNumber_Xor');
  @PyOS_InitInterrupts       :=Import('PyOS_InitInterrupts');
  @PyOS_InterruptOccurred    :=Import('PyOS_InterruptOccurred');
  @PyObject_CallObject       :=Import('PyObject_CallObject');
  @PyObject_Compare          :=Import('PyObject_Compare');
  @PyObject_GetAttr          :=Import('PyObject_GetAttr');
  @PyObject_GetAttrString    :=Import('PyObject_GetAttrString');
  @PyObject_GetItem          :=Import('PyObject_GetItem');
  @PyObject_HasAttrString    :=Import('PyObject_HasAttrString');
  @PyObject_Hash             :=Import('PyObject_Hash');
  @PyObject_IsTrue           :=Import('PyObject_IsTrue');
  @PyObject_Length           :=Import('PyObject_Length');
  @PyObject_Repr             :=Import('PyObject_Repr');
  @PyObject_SetAttr          :=Import('PyObject_SetAttr');
  @PyObject_SetAttrString    :=Import('PyObject_SetAttrString');
  @PyObject_SetItem          :=Import('PyObject_SetItem');
  @PyRange_New               :=Import('PyRange_New');
  @PySequence_Check          :=Import('PySequence_Check');
  @PySequence_Concat         :=Import('PySequence_Concat');
  @PySequence_Count          :=Import('PySequence_Count');
  @PySequence_GetItem        :=Import('PySequence_GetItem');
  @PySequence_GetSlice       :=Import('PySequence_GetSlice');
  @PySequence_In             :=Import('PySequence_In');
  @PySequence_Index          :=Import('PySequence_Index');
  @PySequence_Length         :=Import('PySequence_Length');
  @PySequence_Repeat         :=Import('PySequence_Repeat');
  @PySequence_SetItem        :=Import('PySequence_SetItem');
  @PySequence_SetSlice       :=Import('PySequence_SetSlice');
  @PySequence_Tuple          :=Import('PySequence_Tuple');
  @PySlice_GetIndices        :=Import('PySlice_GetIndices');
  @PySlice_New               :=Import('PySlice_New');
  @PyString_Concat           :=Import('PyString_Concat');
  @PyString_ConcatAndDel     :=Import('PyString_ConcatAndDel');
  @PyString_Format           :=Import('PyString_Format');
  @PyString_FromStringAndSize:=Import('PyString_FromStringAndSize');
  @PyString_Size             :=Import('PyString_Size');
  @PySys_GetObject           :=Import('PySys_GetObject');
  //@PySys_Init                :=Import('PySys_Init');
  @PySys_SetObject           :=Import('PySys_SetObject');
  @PySys_SetPath             :=Import('PySys_SetPath');
  //@PyTraceBack_Fetch         :=Import('PyTraceBack_Fetch');
  @PyTraceBack_Here          :=Import('PyTraceBack_Here');
  @PyTraceBack_Print         :=Import('PyTraceBack_Print');
  //@PyTraceBack_Store         :=Import('PyTraceBack_Store');
  @PyTuple_GetItem           :=Import('PyTuple_GetItem');
  @PyTuple_GetSlice          :=Import('PyTuple_GetSlice');
  @PyTuple_New               :=Import('PyTuple_New');
  @PyTuple_SetItem           :=Import('PyTuple_SetItem');
  @PyTuple_Size              :=Import('PyTuple_Size');
  @Py_AtExit                 :=Import('Py_AtExit');
  //@Py_Cleanup                :=Import('Py_Cleanup');
  @Py_CompileString          :=Import('Py_CompileString');
  @Py_FatalError             :=Import('Py_FatalError');
  @Py_FindMethod             :=Import('Py_FindMethod');
  @Py_FindMethodInChain      :=Import('Py_FindMethodInChain');
  @Py_FlushLine              :=Import('Py_FlushLine');
  @_PyObject_New             :=Import('_PyObject_New');
  @_PyString_Resize          :=Import('_PyString_Resize');
  //@exit_thread               :=Import('exit_thread');
  //@get_thread_ident          :=Import('get_thread_ident');
  //@init_thread               :=Import('init_thread');
  //@start_new_thread          :=Import('start_new_thread');
  @Py_Finalize                :=Import('Py_Finalize');
  if getProcAddress( FDLLHandle, 'PyCode_Addr2Line' ) <> nil then
    @DLL_PyCode_Addr2Line     := Import('PyCode_Addr2Line');
  @PyClass_IsSubclass         :=Import('PyClass_IsSubclass');
  @PyErr_ExceptionMatches     :=Import('PyErr_ExceptionMatches');
  @PyErr_GivenExceptionMatches:=Import('PyErr_GivenExceptionMatches');
  @PyEval_EvalCode            :=Import('PyEval_EvalCode');
  @Py_GetVersion              :=Import('Py_GetVersion');
  @Py_GetCopyright            :=Import('Py_GetCopyright');
  @PyParser_SimpleParseString :=Import('PyParser_SimpleParseString');
  @PyNode_Free                :=Import('PyNode_Free');
  @PyErr_NewException         :=Import('PyErr_NewException');
  @Py_Malloc                  :=Import('Py_Malloc');
  @PyObject_CallMethod        :=Import('PyObject_CallMethod');
end;

procedure TPythonInterface.Py_INCREF(op: PPyObject);
begin
  Inc(op^.ob_refcnt);
end;

procedure TPythonInterface.Py_DECREF(op: PPyObject);
begin
  with op^ do begin
    Dec(ob_refcnt);
    if ob_refcnt = 0 then begin
      ob_type^.tp_dealloc(op);
    end;
  end;
end;

procedure TPythonInterface.Py_XINCREF(op: PPyObject);
begin
  if op <> nil then Py_INCREF(op);
end;

procedure TPythonInterface.Py_XDECREF(op: PPyObject);
begin
  if op <> nil then Py_DECREF(op);
end;

function TPythonInterface.Py_BuildValue( format: PChar; args: array of const):
         PPyObject; assembler; cdecl;
const tenthousand:double = 10000.0;
var temp: Double;
asm
        push ebx             // save ebx, four registers needed
        mov eax,[args]       // gets args pointer
        mov edx,[args+$4]    // gets invisible argument count-1 parameter
@loop1: lea ebx,[eax+edx*8]  // get argument address
        cmp byte ptr [ebx+$4], vtExtended // Is Extended?
        jz  @IsDouble
        cmp byte ptr [ebx+$4], vtCurrency // Is Currency, 64bit integer?
        jnz @NoDouble
@IsCurrency:
        mov ebx,[ebx]        // get 64 bit integer
        fild qword ptr [ebx] // put on float stack
        wait
        fdiv tenthousand     // normalize to double float
        jmp @PushDouble      // Handle as double value
@IsDouble:
        mov ebx,[ebx]        // get extended float
        fld tbyte ptr [ebx]  // put on float stack
@PushDouble:
        wait
        fstp temp            //get double from float stack
        wait
        lea ebx,temp         // get temp
        mov ecx,[ebx+$4]     // push 4 high bytes of temp
        push ecx
@NoDouble:
        mov ecx,[ebx]        // get one argument
        push ecx             // push 4 bytes of the argument
        dec edx              // go to the next argument
        jns  @loop1          // next argument, if any
        mov  eax,self        // get invisible self parameter
        mov  edx,format      // call DLL Py_Builtin function
        push edx
        call [eax+DLL_Py_BuildValue]
        pop  ebx             // Dummy pop of result high 4 bytes
        pop  ebx             // real pop, get saved ebx
end;

{DELPHI does the right thing here. It automatically generates
 a copy of the argp on the stack}
function TPythonInterface.PyArg_Parse ( args: PPyObject; format: PChar;
                       argp: array of Pointer): Integer; cdecl;
begin
  PyArg_Parse := DLL_PyArg_Parse(args, format);
end;

function TPythonInterface.PyArg_ParseTuple ( args: PPyObject; format: PChar;
                            argp: array of Pointer): Integer; cdecl;
begin
  PyArg_ParseTuple := DLL_PyArg_ParseTuple(args, format);
end;

// This function is copied from compile.c because it was not
// exported in the Dll
function TPythonInterface.PyCode_Addr2Line( co: PPyCodeObject; addrq : Integer ) : Integer; cdecl;
var
  size : Integer;
  p    : PChar;
  line : Integer;
  addr : Integer;
  cpt  : Integer;
begin
  if Assigned(DLL_PyCode_Addr2Line) then
    begin
      Result := DLL_PyCode_Addr2Line( co, addrq );
      Exit;
    end;
  size := PyString_Size(co^.co_lnotab) div 2;
  p    := PyString_AsString(co^.co_lnotab);
  line := co^.co_firstlineno;
  addr := 0;
  cpt  := 0;
  while (size-1) >= 0 do
    begin
      Dec(size);
      Inc( addr, Ord(p[cpt]) );
      Inc(cpt);
      if addr > addrq then
        Break;
      Inc( line, Ord(p[cpt]) );
      Inc(cpt);
    end;
  Result := line;
end;

function TPythonInterface.PyImport_ExecCodeModule( const name : String; codeobject : PPyObject) : PPyObject;
var
  m, d, v, modules : PPyObject;
begin
  CheckPython;
  m := PyImport_AddModule(PChar(name));
  if not Assigned(m) then
    begin
      Result := nil;
      Exit;
    end;
  d := PyModule_GetDict(m);
  if PyDict_GetItemString(d, '__builtins__') = nil then
    begin
      if PyDict_SetItemString(d, '__builtins__', PyEval_GetBuiltins) <> 0 then
        begin
          Result := nil;
          Exit;
        end;
    end;
  // Remember the fielname as the __file__ attribute
  if PyDict_SetItemString(d, '__file__', PPyCodeObject(codeobject)^.co_filename) <> 0 then
    PyErr_Clear; // Not important enough to report
  v := PyEval_EvalCode(PPyCodeObject(codeobject), d, d); // XXX owner ?
  if not Assigned(v) then
    begin
      Result := nil;
      Exit;
    end;
  Py_XDECREF(v);
  modules := PyImport_GetModuleDict;
  if PyDict_GetItemString(modules, PChar(name)) = nil then
    begin
      PyErr_SetString(PyExc_ImportError^, PChar(Format('Loaded module %.200s not found in sys.modules', [name])));
      Result := nil;
      Exit;
    end;
  Py_XINCREF(m);
  Result := m;
end;

function TPythonInterface.PyString_Check( obj : PPyObject ) : Boolean;
begin
  Result := Assigned( obj ) and (obj^.ob_type = PPyTypeObject(PyString_Type));
end;

function TPythonInterface.PyFloat_Check( obj : PPyObject ) : Boolean;
begin
  Result := Assigned( obj ) and (obj^.ob_type = PPyTypeObject(PyFloat_Type));
end;

function TPythonInterface.PyInt_Check( obj : PPyObject ) : Boolean;
begin
  Result := Assigned( obj ) and (obj^.ob_type = PPyTypeObject(PyInt_Type));
end;

function TPythonInterface.PyTuple_Check( obj : PPyObject ) : Boolean;
begin
  Result := Assigned( obj ) and (obj^.ob_type = PPyTypeObject(PyTuple_Type));
end;

function TPythonInterface.PyInstance_Check( obj : PPyObject ) : Boolean;
begin
  Result := Assigned( obj ) and (obj^.ob_type = PPyTypeObject(PyInstance_Type));
end;

function TPythonInterface.PyClass_Check( obj : PPyObject ) : Boolean;
begin
  Result := Assigned( obj ) and (obj^.ob_type = PPyTypeObject(PyClass_Type));
end;

function TPythonInterface.PyMethod_Check( obj : PPyObject ) : Boolean;
begin
  Result := Assigned( obj ) and (obj^.ob_type = PPyTypeObject(PyMethod_Type));
end;

function TPythonInterface.PyList_Check( obj : PPyObject ) : Boolean;
begin
  Result := Assigned( obj ) and (obj^.ob_type = PPyTypeObject(PyList_Type));
end;

function TPythonInterface.PyDict_Check( obj : PPyObject ) : Boolean;
begin
  Result := Assigned( obj ) and (obj^.ob_type = PPyTypeObject(PyDict_Type));
end;

function TPythonInterface.PyModule_Check( obj : PPyObject ) : Boolean;
begin
  Result := Assigned( obj ) and (obj^.ob_type = PPyTypeObject(PyModule_Type));
end;

function TPythonInterface.PyFunction_Check( obj : PPyObject ) : Boolean;
begin
  Result := Assigned( obj ) and
    ((obj^.ob_type = PPyTypeObject(PyCFunction_Type)) or
     (obj^.ob_type = PPyTypeObject(PyFunction_Type)));
end;

function TPythonInterface.Py_InitModule( const name : PChar; md : PPyMethodDef) : PPyObject;
begin
  CheckPython;
  result := Py_InitModule4( name, md, nil, nil, APIVersion );
end;


(*******************************************************)
(**                                                   **)
(**            class TPythonEngine                    **)
(**                                                   **)
(*******************************************************)


var
  gPythonEngine : TPythonEngine;

constructor TPythonEngine.Create(AOwner: TComponent);
var
  i : Integer;
begin
  inherited;
  InitializeCriticalSection( FLock );
  FInitialized         := False;
  FInitScript          := TstringList.Create;
  FClients             := TList.Create;
  FRedirectIO          := True;
  FExecModule          := '__main__';
  FAutoFinalize        := True;
  if csDesigning in ComponentState then
    begin
      for i := 0 to AOwner.ComponentCount - 1 do
        if (AOwner.Components[i] is TPythonEngine) and
           (AOwner.Components[i] <> Self) then
          raise Exception.Create('You can''t drop more than one TPythonEngine component');
    end
  else
    if Assigned(gPythonEngine) then
      raise Exception.Create('There is already one instance of TPythonEngine running' );
  gPythonEngine        := Self;
end;

destructor TPythonEngine.Destroy;
var
  i: integer;
begin
  for i := 0 to ClientCount - 1 do
    with Clients[i] do
      begin
        if Initialized then
          Finalize;
        ClearEngine;
      end;
  FClients.Free;
  FInitScript.Free;
  if FInitialized and FAutoFinalize then
    Py_Finalize;
  gPythonEngine := nil;
  DeleteCriticalSection( FLock );
  inherited;
end;

procedure TPythonEngine.Lock;
begin
  EnterCriticalSection(FLock);
end;

procedure TPythonEngine.Unlock;
begin
  LeaveCriticalSection(FLock);
end;

procedure TPythonEngine.AfterLoad;
begin
  inherited;
  Initialize;
end;

procedure TPythonEngine.Initialize;
var
  i : Integer;
begin
  Py_Initialize;
  FInitialized := True;
  if RedirectIO and Assigned(FIO) then
    DoRedirectIO;
  for i := 0 to ClientCount - 1 do
    with Clients[i] do
      if not Initialized then
        Initialize;
  if InitScript.Count > 0 then
    ExecStrings( InitScript );
  if Assigned(FOnAfterInit) then
    FOnAfterInit(Self);
end;

procedure TPythonEngine.SetInitScript(Value: TStrings);
begin
  FInitScript.Assign(Value);
end;

function TPythonEngine.GetClientCount : Integer;
begin
  Result := FClients.Count;
end;

function TPythonEngine.GetClients( idx : Integer ) : TEngineClient;
begin
  Result := TEngineClient( FClients.Items[idx] );
end;

procedure TPythonEngine.Notification( AComponent: TComponent;
                                      Operation: TOperation);
begin
  if Operation = opRemove then
    if AComponent = IO then
      IO := nil;
end;

function TPythonEngine.IsType(ob: PPyObject; obt: PPyTypeObject): Boolean;
begin
  result := ob^.ob_type = obt;
end;

function TPythonEngine.GetAttrString(obj: PPyObject; name: PChar):PChar;
var
   attr: PPyObject;
begin
  CheckPython;
  result := nil;
  attr := PyObject_GetAttrString(obj,name);
  if attr <> nil then begin
    result := PyString_AsString(attr);
    Py_XDECREF(attr);
  end;
  PyErr_Clear;
end;

function TPythonEngine.CleanString(const s : string) : string;
var
  i : Integer;
begin
  result := s;
  if s = '' then
    Exit;
  i := Pos(CR,s);
  while i > 0 do
    begin
      Delete( result, i, 1 );
      i := Pos(CR,result);
    end;
  if result[length(result)] <> LF then
    Insert( LF, result, length(result)+1 );
end;

function   TPythonEngine.EvalPyFunction(pyfunc, pyargs:PPyObject): Variant;
var presult :PPyObject;
begin
  CheckPython;
  result := -1;
  if pyfunc = nil then exit;
  try
    presult := PyEval_CallObject(pyfunc,pyargs);
    if presult = nil then
      begin
        PyErr_Print;
        RaiseError;
      end
    else
      begin
        if presult = Py_None then result := 0
        else result := PyObjectAsVariant( presult );
        Py_DECREF(presult);
      end;
    Py_FlushLine;
  except
    Py_FlushLine;
    if PyErr_Occurred <> nil then
      begin
        PyErr_Print;
        RaiseError;
      end
    else
      raise;
  end;
end;

function   TPythonEngine.EvalFunction(pyfunc:PPyObject; args: array of const): Variant;
var pargs: PPyObject;
begin
  CheckPython;
  pargs := ArrayToPyTuple(args);
  result := EvalPyFunction(pyfunc,pargs);
  Py_DECREF(pargs);
end;

function   TPythonEngine.EvalFunctionNoArgs(pyfunc:PPyObject): Variant;
var pargs: PPyObject;
begin
  CheckPython;
  pargs := PyTuple_New(0);
  result := EvalPyFunction(pyfunc, pargs);
  Py_DECREF(pargs);
end;

function   TPythonEngine.EvalStringAsStr(const command : String) : String;
begin
  Result := Run_CommandAsString( command, eval_input );
end;

function   TPythonEngine.EvalString(const command : String) : PPyObject;
begin
  Result := Run_CommandAsObject( command, eval_input );
end;

procedure TPythonEngine.ExecString(const command : String);
begin
  Run_CommandAsString( command, file_input );
end;

function   TPythonEngine.Run_CommandAsString(const command : String; mode : Integer) : String;
var
  v : PPyObject;
begin
  Result := '';
  v := Run_CommandAsObject( command, mode );
  Result := PyObjectAsString( v );
  Py_XDECREF(v);
end;

function   TPythonEngine.Run_CommandAsObject(const command : String; mode : Integer) : PPyObject;
var
  m, d : PPyObject;
begin
  CheckPython;
  Result := nil;
  m := PyImport_AddModule(PChar(ExecModule));
  if m = nil then
    raise EPythonError.Create('Run_CommandAsObject: can''t create __main__');
  d := PyModule_GetDict(m);
  try
    Result := PyRun_String(PChar(CleanString(command)), mode, d, d);
    if Result = nil then
      begin
        PyErr_Print;
        RaiseError;
      end;
    Py_FlushLine;
  except
    Py_FlushLine;
    if PyErr_Occurred <> nil then
      begin
        PyErr_Print;
        RaiseError;
      end
    else
      raise;
  end;
end;

procedure TPythonEngine.ExecStrings( strings : TStrings );
begin
  Py_XDecRef( Run_CommandAsObject( CleanString( strings.Text ), file_input ) );
end;

function TPythonEngine.EvalStrings( strings : TStrings ) : PPyObject;
begin
  Result := Run_CommandAsObject( CleanString( strings.Text ), eval_input );
end;

function TPythonEngine.EvalStringsAsStr( strings : TStrings ) : String;
begin
  Result := Run_CommandAsString( CleanString( strings.Text ), eval_input );
end;

function TPythonEngine.CheckEvalSyntax( const str : String ) : Boolean;
begin
  result := CheckSyntax( str, eval_input );
end;

function TPythonEngine.CheckExecSyntax( const str : String ) : Boolean;
begin
  result := CheckSyntax( str, file_input );
end;

function TPythonEngine.CheckSyntax( const str : String; mode : Integer ) : Boolean;
var
  n : PNode;
begin
  n := PyParser_SimpleParseString( PChar(str), mode );
  result := Assigned(n);
  if Assigned( n ) then
    PyNode_Free(n);
end;

procedure TPythonEngine.RaiseError;

  function Define( E : EPythonError; const sType, sValue : String ) : EPythonError;
  begin
    E.EName  := sType;
    E.EValue := sValue;
    if sValue <> '' then
      E.Message := Format('%s: %s',[sType,sValue])
    else
      E.Message := sType;
    Result := E;
  end;

  function GetTypeAsString( obj : PPyObject ) : String;
  begin
    if PyClass_Check( obj ) then
      with PPyClassObject(obj)^ do
        Result := StrPas( PyString_AsString(cl_name) )
    else
      Result := PyObjectAsString(obj);
  end;

var
  err_type, err_value : PPyObject;
  s_type        : String;
  s_value       : String;
  s_line        : String;
  s_filename    : String;
  i_line_number : Integer;
  i_offset      : Integer;
  syntax        : EPySyntaxError;
  d, tmp        : PPyObject;
begin
  s_value       := '';
  s_line        := '';
  s_filename    := '';
  i_line_number := 0;
  i_offset      := 0;

  if PyErr_Occurred <> nil then
    PyErr_Print;
  err_type := PySys_GetObject('last_type');
  err_value := PySys_GetObject('last_value');
  if Assigned(err_type) then
    begin
      s_type := GetTypeAsString(err_type);
      s_value := PyObjectAsString(err_value);
      // Is it a syntax error ?
      if (PyErr_GivenExceptionMatches(err_type, PyExc_SyntaxError^) <> 0) then
        begin
          s_value := '';
          // Is it an instance of the SyntaxError class ?
          if PyInstance_Check( err_value ) and
             (PyClass_IsSubclass( PPyObject(PPyInstanceObject(err_value)^.in_class), err_type ) <> 0) then
            begin
              // Get informations from the instance object
              d := PPyInstanceObject(err_value)^.in_dict;
              tmp := PyDict_GetItemString( d, 'filename' );
              // Get the filename
              if Assigned(tmp) then
                if PyString_Check(tmp) then
                  s_filename := PyString_AsString(tmp)
                else if tmp = Py_None then
                  s_filename := '???';
              // Get the text containing the error
              tmp := PyDict_GetItemString( d, 'text' );
              if Assigned(tmp) and PyString_Check(tmp) then
                s_line := PyString_AsString(tmp);
              // Get the offset where the error should appear
              tmp := PyDict_GetItemString( d, 'offset' );
              if Assigned(tmp) and PyInt_Check(tmp) then
                i_offset := PyInt_AsLong(tmp);
              // Get the line number of the error
              tmp := PyDict_GetItemString( d, 'lineno' );
              if Assigned(tmp) and PyInt_Check(tmp) then
                i_line_number := PyInt_AsLong(tmp);
              // Get the message of the error
              tmp := PyDict_GetItemString( d, 'msg' );
              if Assigned(tmp) and PyString_Check(tmp) then
                s_value := PyString_AsString(tmp);
            end;
          // If all is ok
          if s_value <> '' then
            begin
              // Build a Syntax error exception with all informations
              syntax := EPySyntaxError.CreateFmt('%s: %s (line %d, offset %d)',[s_type,s_value,i_line_number, i_offset]);
              with syntax do
                begin
                  EName       := s_type;
                  EValue      := s_value;
                  EFileName   := s_filename;
                  ELineNumber := i_line_number;
                  EOffset     := i_offset;
                  ELineStr    := s_line;
                end;
            end
          else
            syntax := EPySyntaxError.Create(s_type);
          raise syntax;
        end
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_AttributeError^) <> 0) then
        raise Define( EPyAttributeError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_EOFError^) <> 0) then
        raise Define( EPyEOFError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_IOError^) <> 0) then
        raise Define( EPyIOError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_ImportError^) <> 0) then
        raise Define( EPyImportError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_IndexError^) <> 0) then
        raise Define( EPyIndexError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_KeyError^) <> 0) then
        raise Define( EPyKeyError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_KeyboardInterrupt^) <> 0) then
        raise Define( EPyKeyboardInterrupt.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_MemoryError^) <> 0) then
        raise Define( EPyMemoryError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_NameError^) <> 0) then
        raise Define( EPyNameError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_OverflowError^) <> 0) then
        raise Define( EPyOverflowError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_RuntimeError^) <> 0) then
        raise Define( EPyRuntimeError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_SystemError^) <> 0) then
        raise Define( EPySystemError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_SystemExit^) <> 0) then
        raise Define( EPySystemExit.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_TypeError^) <> 0) then
        raise Define( EPyTypeError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_ValueError^) <> 0) then
        raise Define( EPyValueError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_ZeroDivisionError^) <> 0) then
        raise Define( EPyZeroDivisionError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_Exception^) <> 0) then
        raise Define( EPyException.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_FloatingPointError^) <> 0) then
        raise Define( EPyFloatingPointError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_LookupError^) <> 0) then
        raise Define( EPyLookupError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_StandardError^) <> 0) then
        raise Define( EPyStandardError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_AttributeError^) <> 0) then
        raise Define( EPyAttributeError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_AttributeError^) <> 0) then
        raise Define( EPyAttributeError.Create(''), s_type, s_value )
      else if (PyErr_GivenExceptionMatches(err_type, PyExc_AttributeError^) <> 0) then
        raise Define( EPyAttributeError.Create(''), s_type, s_value )
      else  // Else if no known exception was detected,
            // then build an ExecError exception
        raise Define( EPyExecError.Create(''), s_type, s_value );
    end
  else
    raise EPythonError.Create('RaiseError: could''nt fetch last exception');
end;

function TPythonEngine.PyObjectAsString( obj : PPyObject ) : String;
var
  s : PPyObject;
  i : Integer;
  tmp : PChar;
begin
  CheckPython;
  Result := '';
  if not Assigned( obj ) then
    Exit;

  s := PyObject_Str( obj );
  if Assigned(s) and PyString_Check(s) then
    begin
      tmp := PyString_AsString(s);
      SetLength( Result, PyString_Size(s)+1 );
      Result := '';
      for i := 0 to PyString_Size(s) - 1 do
        Insert( tmp[i], Result, i+1 );
    end;
  Py_XDECREF(s);
end;

procedure TPythonEngine.DoRedirectIO;
const
  code = 'import sys'+LF+
         'import pyio'+LF+
         'class DebugOutput:'+LF+
         '  softspace=1'+LF+
         '  def write(self,message):'+LF+
         '     pyio.write(message)'+LF+
         '  def readline(size):'+LF+
         '     return pyio.read(size)'+LF+
         'sys.stdin=sys.stderr=sys.stdout=DebugOutput()'+LF+#0;
begin
  if csDesigning in ComponentState then
    Exit;
  CheckPython;
  // create a new module called pyio
  with TPythonModule.Create( Owner ) do
    begin
      Engine := Self;
      ModuleName := 'pyio';
      AddMethod( 'write', pyio_write, 'write(String) -> None' );
      AddMethod( 'read',  pyio_read,  'read() -> String' );
      AddMethod( 'SetDelayWrites',  pyio_SetDelayWrites,  'SetDelayWrites(Boolean) -> None' );
      AddMethod( 'SetMaxLines',  pyio_SetMaxLines,  'SetMaxLines(Integer) -> None' );
      Initialize;
    end;
  // execute the code
  ExecString(code);
end;

procedure  TPythonEngine.AddClient( client : TEngineClient );
begin
  FClients.Add( client );
end;

procedure  TPythonEngine.RemoveClient( client : TEngineClient );
begin
  if client.Initialized then
    client.Finalize;
  FClients.Remove( client );
end;

function   TPythonEngine.TypeByName( const aTypeName : String ) : PPyTypeObject;
var
  i : Integer;
begin
  for i := 0 to ClientCount - 1 do
    if Clients[i] is TPythonType then
      with TPythonType( Clients[i] ) do
        if TypeName = aTypeName then
          begin
            Result := TheTypePtr;
            Exit;
          end;
  raise Exception.CreateFmt('Could not find type: %s', [aTypeName]);
end;

function   TPythonEngine.ModuleByName( const aModuleName : String ) : PPyObject;
var
  i : Integer;
begin
  for i := 0 to ClientCount - 1 do
    if Clients[i] is TPythonModule then
      with TPythonModule( Clients[i] ) do
        if ModuleName = aModuleName then
          begin
            Result := Module;
            Exit;
          end;
  raise Exception.CreateFmt('Could not find module: %s', [aModuleName]);
end;

function   TPythonEngine.MethodsByName( const aMethodsContainer: String ) : PPyMethodDef;
var
  i : Integer;
begin
  for i := 0 to ClientCount - 1 do
    if Clients[i] is TMethodsContainer then
      with TMethodsContainer( Clients[i] ) do
        if Name = aMethodsContainer then
          begin
            Result := MethodsData;
            Exit;
          end;
  raise Exception.CreateFmt('Could not find component: %s', [aMethodsContainer]);
end;

function TPythonEngine.VariantAsPyObject( V : Variant ) : PPyObject;
var
  s : String;
begin
  case VarType(V) of
    varBoolean,
    varSmallint,
    varInteger:  Result := PyInt_FromLong( V );
    varSingle,
    varDouble,
    varCurrency: Result := PyFloat_FromDouble( V );
    varOleStr,
    varString:
      begin
        s := V;
        Result := PyString_FromString( PChar(s) );
      end;
  else
    Result := ReturnNone;
  end;
end;

function TPythonEngine.PyObjectAsVariant( obj : PPyObject ) : Variant;
var
  s : PChar;
begin
  if PyInt_Check(obj) then
    Result := PyInt_AsLong( obj )
  else if PyFloat_Check(obj) then
    Result := PyFloat_AsDouble(obj)
  else if PyString_Check(obj) then
    begin
      s := PyString_AsString( obj );
      Result := Copy(s, 1, strlen(s));
    end
  else
    Result := varNull;
end;

function TPythonEngine.VarRecAsPyObject( v : TVarRec ) : PPyObject;
var
  buff : array [0..256] of Char;
begin
  case v.VType of
    vtInteger:       Result := PyInt_FromLong( v.VInteger );
    vtBoolean:       Result := PyInt_FromLong( Integer(v.VBoolean) );
    vtChar:          Result := PyInt_FromLong( Integer(v.VChar) );
    vtExtended:      Result := PyFloat_FromDouble( v.VExtended^ );
    vtString:        Result := PyString_FromString( StrPCopy( buff, v.VString^) );
    vtPChar:         Result := PyString_FromString( v.VPChar );
    vtAnsiString:    Result := PyString_FromString( v.VAnsiString );
    vtCurrency:      Result := PyFloat_FromDouble( v.VCurrency^ );
    vtVariant:       Result := VariantAsPyObject( v.VVariant^ );
    vtPointer:       Result := v.VPointer;
  else
    Raise Exception.Create('Argument type not allowed');
  end;
end;

// This function prevents Python from deleting the objects contained
// when the container will be freed, because we increment each
// object's refcount.
function TPythonEngine.MakePyTuple( const objects : array of PPyObject ) : PPyObject;
var
  i : Integer;
begin
  Result := PyTuple_New( High(objects)+1 );
  if not Assigned(Result) then
    raise EPythonError.Create('Could not create a new tuple object');
  for i := Low(objects) to High(objects) do
    begin
      Py_XINCREF( objects[i] );
      PyTuple_SetItem( Result, i, objects[i] );
    end;
end;

// This function prevents Python from deleting the objects contained
// when the container will be freed, because we increment each
// object's refcount.
function TPythonEngine.MakePyList( const objects : array of PPyObject ) : PPyObject;
var
  i : Integer;
begin
  Result := PyList_New( High(objects)+1 );
  if not Assigned(Result) then
    raise EPythonError.Create('Could not create a new list object');
  for i := Low(objects) to High(objects) do
    begin
      Py_XINCREF( objects[i] );
      PyList_SetItem( Result, i, objects[i] );
    end;
end;

function TPythonEngine.ArrayToPyTuple( items : array of const) : PPyObject;
var
  i : Integer;
begin
  Result := PyTuple_New( High(items)+1 );
  if not Assigned(Result) then
    raise EPythonError.Create('Could not create a new tuple object');
  for i := Low(items) to High(items) do
    PyTuple_SetItem( Result, i, VarRecAsPyObject( items[i] ) );
end;

function TPythonEngine.ArrayToPyList( items : array of const) : PPyObject;
var
  i : Integer;
begin
  Result := PyList_New( High(items)+1 );
  if not Assigned(Result) then
    raise EPythonError.Create('Could not create a new list object');
  for i := Low(items) to High(items) do
    PyList_SetItem( Result, i, VarRecAsPyObject( items[i] ) );
end;

// You must give each entry as a couple key(string)/value
function TPythonEngine.ArrayToPyDict( items : array of const) : PPyObject;

  function VarRecAsString( v : TVarRec ) : String;
  begin
    case v.VType of
      vtChar:          Result := v.VChar;
      vtString:        Result := v.VString^;
      vtPChar:         Result := v.VPChar;
      vtAnsiString:    Result := StrPas(v.VAnsiString);
      vtVariant:       Result := v.VVariant^;
    else
      Raise Exception.Create('Argument type not allowed');
    end;
  end;

var
  i : Integer;
  s : String;
begin
  if ((High(items)+1) mod 2) <> 0 then
    raise Exception.Create('You must provide an even number of arguments');
  Result := PyDict_New;
  if not Assigned(Result) then
    raise EPythonError.Create('Could not create a new dict object');
  i := Low(items);
  try
    while i <= High(items) do
      begin
        s := VarRecAsString( items[i] );
        PyDict_SetItemString( Result, PChar(s), VarRecAsPyObject( items[i+1] ) );
        Inc( i, 2 );
      end;
  except
    Py_XDECREF( Result );
  end;
end;

function TPythonEngine.StringsToPyList( strings : TStrings ) : PPyObject;
var
  i : Integer;
begin
  Result := PyList_New( strings.Count );
  if not Assigned(Result) then
    raise EPythonError.Create('Could not create a new list object');
  for i := 0 to strings.Count - 1 do
    PyList_SetItem( Result, i, PyString_FromString( PChar(strings.Strings[i]) ) );
end;

function TPythonEngine.StringsToPyTuple( strings : TStrings ) : PPyObject;
var
  i : Integer;
begin
  Result := PyTuple_New( strings.Count );
  if not Assigned(Result) then
    raise EPythonError.Create('Could not create a new tuple object');
  for i := 0 to strings.Count - 1 do
    PyTuple_SetItem( Result, i, PyString_FromString( PChar(strings.Strings[i]) ) );
end;

procedure TPythonEngine.PyListToStrings( list : PPyObject; strings : TStrings );
var
  i : Integer;
begin
  if not PyList_Check(list) then
    raise EPythonError.Create('the python object is not a list');
  strings.Clear;
  for i := 0 to PyList_Size( list ) - 1 do
    strings.Add( PyObjectAsString( PyList_GetItem( list, i ) ) );
end;

procedure TPythonEngine.PyTupleToStrings( tuple: PPyObject; strings : TStrings );
var
  i : Integer;
begin
  if not PyTuple_Check(tuple) then
    raise EPythonError.Create('the python object is not a tuple');
  strings.Clear;
  for i := 0 to PyTuple_Size( tuple ) - 1 do
    strings.Add( PyObjectAsString( PyTuple_GetItem( tuple, i ) ) );
end;

function TPythonEngine.ReturnNone : PPyObject;
begin
  Result := Py_None;
  Py_INCREF( Result );
end;

function TPythonEngine.FindModule( const ModuleName : String ) : PPyObject;
var
  modules, m : PPyObject;
begin
  modules := PyImport_GetModuleDict;
  m := PyDict_GetItemString(modules, PChar(ModuleName) );
  if (m <> nil) and (PyModule_Check(m)) then
    Result := m
  else
    Result := nil;
end;

function TPythonEngine.FindFunction(ModuleName,FuncName: String): PPyObject;
var
  module,func: PPyObject;
begin
  module := FindModule(ModuleName);
  if module = nil then result := nil
  else begin
    func := PyObject_GetAttrString(module, PChar(FuncName));
    if (func <> nil) and (PyFunction_Check(func)) then
      Result := func
    else
      Result := nil;
  end;
end;

(*******************************************************)
(**                                                   **)
(**     class TEngineClient                           **)
(**                                                   **)
(*******************************************************)

procedure  TEngineClient.SetEngine( val : TPythonEngine );
begin
  if val <> FEngine then
    begin
      if Assigned(FEngine) {and not(csDesigning in ComponentState)} then
        FEngine.RemoveClient( Self );
      FEngine := val;
      if Assigned(FEngine) {and not(csDesigning in ComponentState)} then
        FEngine.AddClient( Self );
    end;
end;

constructor TEngineClient.Create( AOwner : TComponent );
begin
  inherited;
end;

destructor TEngineClient.Destroy;
begin
  Engine := nil; // This causes to be detached from the Engine.
                 // And if we were initialized, the Engine finalize us.
  if Assigned( FOnDestroy ) then
    FOnDestroy( Self );
  inherited;
end;

procedure TEngineClient.Loaded;
begin
  inherited;
  if Assigned( FOnCreate ) then
    FOnCreate( Self );
end;

procedure  TEngineClient.Initialize;
begin
  if Assigned( FOnInitialization ) then
     FOnInitialization( Self );
  FInitialized := True;
end;

procedure TEngineClient.Finalize;
begin
  if Assigned( FOnFinalization ) then
     FOnFinalization( Self );
  FInitialized := False;
end;

procedure  TEngineClient.ClearEngine;
begin
  FEngine := nil;
end;

procedure  TEngineClient.CheckEngine;
begin
  if not Assigned(FEngine) then
    raise Exception.CreateFmt('No Engine defined for component "%s"', [Name]);
end;


(*******************************************************)
(**                                                   **)
(**     class TMethodsContainer                       **)
(**                                                   **)
(*******************************************************)

procedure TMethodsContainer.AllocMethods;
begin
  FAllocatedMethodCount := PYT_METHOD_BUFFER_INCREASE;
  FMethods := PPyMethodDef(AllocMem(SizeOf(PyMethodDef)*(FAllocatedMethodCount+1)));
end;

procedure TMethodsContainer.FreeMethods;
begin
  FreeMem(FMethods);
end;

procedure TMethodsContainer.ReallocMethods;
var
  MethodPtr : PPyMethodDef;
begin
  Inc( FAllocatedMethodCount, PYT_METHOD_BUFFER_INCREASE );
  ReAllocMem( FMethods, SizeOf(PyMethodDef)*(FAllocatedMethodCount+1));
  MethodPtr :=@(PMethodArray(FMethods)^[MethodCount+1]);
  FillChar( MethodPtr^,SizeOf(PyMethodDef)*PYT_METHOD_BUFFER_INCREASE,0);
end;

function TMethodsContainer.GetMethods( idx : Integer ) : PPyMethodDef;
begin
  if (idx < 0) or (idx > MethodCount) then
    raise Exception.CreateFmt('%s: Index %d out of range', [ClassName, idx]);
  Result := @( PMethodArray(FMethods)^[idx] );
end;

constructor TMethodsContainer.Create( AOwner : TComponent );
begin
  inherited;
  AllocMethods;
end;

destructor  TMethodsContainer.Destroy;
begin
  FreeMethods;
  inherited;
end;

procedure TMethodsContainer.AddMethod( MethodName  : PChar;
                                       Method  : PyCFunction;
                                       DocString : PChar );
begin
  if FMethodCount = FAllocatedMethodCount then
    ReallocMethods;
  with Methods[ MethodCount ]^ do
    begin
      ml_name  := MethodName;
      ml_meth  := Method;
      ml_flags := 1;
      ml_doc   := DocString;
    end;
  Inc( FMethodCount );
end;


procedure TMethodsContainer.AddDelphiMethod( MethodName  : PChar;
                                             DelphiMethod: TDelphiMethod;
                                             DocString : PChar );
begin
  AddMethod( MethodName,
             GetOfObjectCallBack( TCallBack(DelphiMethod), 2, cdecl),
             DocString );
end;


(*******************************************************)
(**                                                   **)
(**     class TPythonModule                           **)
(**                                                   **)
(*******************************************************)

////////////////////////////////////////
// class TParentClassError

procedure TParentClassError.AssignTo( Dest: TPersistent );
begin
  if Dest is TParentClassError then
     with TParentClassError( Dest ) do
       begin
         FName   := Self.FName;
         FModule := Self.FModule;
       end;
  inherited;
end;

////////////////////////////////////////
// class TError

function TError.GetDisplayName: string;
begin
  Result := Name;
  if Result = '' then Result := inherited GetDisplayName;
end;

procedure TError.SetName( const Value : String );

  procedure CheckName;
  var
    i : Integer;
    m : TPythonModule;
  begin
    with Collection as TErrors do
      begin
        if GetOwner = nil then
          Exit;
        m := GetOwner as TPythonModule;
        for i := 0 to Count - 1 do
          with Items[i] do
            if Name = Value then
              raise Exception.CreateFmt( 'In module "%s", there''s already an error named "%s"',
                                         [m.ModuleName, Value]);
      end;
  end;

  procedure UpdateDependencies;
  var
    i, j : Integer;
    m : TPythonModule;
  begin
    if FName = '' then
      Exit;
    with Collection as TErrors do
      with GetOwner as TPythonModule do
        begin
          if not Assigned(Engine) then
            Exit;
          m := TPythonModule( TErrors(Self.Collection).GetOwner );
          with Engine do
            begin
              for i := 0 to ClientCount - 1 do
                if Clients[i] is TPythonModule then
                  with TPythonModule(Clients[i]) do
                    begin
                      for j := 0 to Errors.Count - 1 do
                        with Errors.Items[j] do
                          if (ParentClass.Module = m.ModuleName) and
                             (ParentClass.Name = Self.Name) then
                            ParentClass.Name := Value;
                    end;
            end;
        end;
  end;

begin
  if (FName <> Value) and (Value <> '') then
  begin
    CheckName;
    if ErrorType = etClass then
      UpdateDependencies;
    FName := Value;
    Changed(False);
  end;
end;

procedure TError.SetText( const Value : String );
begin
  if FText <> Value then
  begin
    FText := Value;
    Changed(False);
  end;
end;

procedure TError.SetErrorType( Value : TErrorType );
begin
  if FErrorType <> Value then
  begin
    FErrorType := Value;
    if FErrorType = etString then
      FParentClass.Name := '';
    Changed(False);
  end;
end;

procedure TError.SetParentClass( Value : TParentClassError );
begin
  FParentClass.Assign( Value );
  Changed(False);
end;

constructor TError.Create(Collection: TCollection);
begin
  inherited;
  FErrorType := etString;
  FParentClass := TParentClassError.Create;
end;

destructor TError.Destroy;
begin
  FParentClass.Free;
  inherited;
end;

procedure TError.Assign(Source: TPersistent);
begin
  if Source is TError then
  begin
    Name := TError(Source).Name;
    Text := TError(Source).Text;
    Exit;
  end;
  inherited Assign(Source);
end;

procedure TError.BuildError( const ModuleName : String );

  function FindParentClass : PPyObject;
  var
    m, d : PPyObject;
  begin
    with GetPythonEngine do
      begin
        if ParentClass.Module <> '' then
          m := PyImport_ImportModule( PChar(ParentClass.Module) )
        else
          m := FindModule( ModuleName );
        if not Assigned(m) then
          raise Exception.CreateFmt('Could not find module containing the parent class of error "%s"', [Self.Name]);
        d := PyModule_GetDict(m);
        Result := PyDict_GetItemString( d, PChar(ParentClass.Name) );
        if not Assigned(Result) then
          raise Exception.CreateFmt('Could not find the parent class "%s" of error "%s"', [ParentClass.Name, Self.Name]);
        if not PyClass_Check( Result ) then
          raise Exception.CreateFmt('The object "%s" in module "%s" is not a class', [ParentClass.Name, ParentClass.Module] );
      end;
  end;

var
  parent : PPyObject;
begin
  if Name = '' then
    with GetOwner as TPythonModule do
      raise Exception.CreateFmt( 'Error without name in module "%s"', [ModuleName] );
  if Text = '' then
    Text := Name;
  with GetPythonEngine do
    begin
      if ErrorType = etString then
        Error := PyString_FromString( PChar(Text) )
      else if ErrorType = etClass then
        begin
          if FParentClass.Name <> '' then
            parent := FindParentClass
          else
            parent := nil;
          Error := PyErr_NewException( PChar(Format('%s.%s', [ModuleName, Self.Name])),
                                       parent, nil );
        end;
    end;
  if not Assigned(Error) then
    raise Exception.CreateFmt( 'Could not create error "%s"', [Name] );
end;

procedure TError.RaiseError( const msg : String );
begin
  with GetPythonEngine do
    PyErr_SetString( Error, PChar(msg) );
end;

procedure TError.RaiseErrorObj( const msg : String; obj : PPyObject );
var
  args, res : PPyObject;
  inst : PPyInstanceObject;
begin
  with GetPythonEngine do
    // if we give a dictionary as argument, then we use it for the
    // instance.
    if PyDict_Check( obj ) then
      begin
        args := PyTuple_New(0);
        if not Assigned(args) then
          raise Exception.Create('TError.RaiseErrorObj: Could not create an empty tuple');
        res := PyEval_CallObject(Error, args);
        Py_DECREF(args);
        if not Assigned(res) then
          raise Exception.CreateFmt('TError.RaiseErrorObj: Could not create an instance of "%s"', [Self.Name]);
        if not PyInstance_Check( res ) then
          raise Exception.Create('TError.RaiseErrorObj: I didn''t get an instance' );
        inst := PPyInstanceObject(res);
        Py_XDECREF( inst^.in_dict );
        inst^.in_dict := obj;
        PyDict_SetItemString( obj, 'args', PyString_FromString( PChar(msg) ) );
        PyErr_SetObject( Error, res );
      end
    else
      PyErr_SetObject( Error, obj );
end;

////////////////////////////////////////
// class TErrors

function TErrors.GetError(Index: Integer): TError;
begin
  Result := TError(inherited GetItem(Index));
end;

procedure TErrors.SetError(Index: Integer; Value: TError);
begin
  inherited SetItem(Index, Value);
end;

function TErrors.GetOwner: TPersistent;
begin
  Result := FModule;
end;

procedure TErrors.Update(Item: TCollectionItem);
begin
  inherited;
end;

constructor TErrors.Create(Module: TPythonModule );
begin
  inherited Create( TError );
  FModule := Module;
end;

function  TErrors.Add: TError;
begin
  Result := TError(inherited Add);
end;

////////////////////////////////////////
// class TPythonModule

function TPythonModule.GetClientCount : Integer;
begin
  Result := FClients.Count;
end;

function TPythonModule.GetClients( idx : Integer ) : TEngineClient;
begin
  Result := TEngineClient(FClients.Items[idx]);
end;

procedure TPythonModule.SetErrors( val : TErrors );
begin
  FErrors.Assign( val );
end;

procedure TPythonModule.SetModuleName( const val : String );

  procedure UpdateDependencies;
  var
    i, j : Integer;
  begin
    if not Assigned(Engine) then
      Exit;
    if FModuleName = '' then
      Exit;
    with Engine do
      for i := 0 to ClientCount - 1 do
        if Clients[i] is TPythonModule then
          with TPythonModule(Clients[i]) do
            for j := 0 to Errors.Count - 1 do
              with Errors.Items[j] do
                if ParentClass.Module = Self.FModuleName then
                  ParentClass.Module := val;
  end;

begin
  if (FModuleName <> val) and (val <> '') then
    begin
      UpdateDependencies;
      FModuleName := val;
    end;
end;

constructor TPythonModule.Create( AOwner : TComponent );
begin
  inherited;
  FClients := TList.Create;
  FErrors  := TErrors.Create(Self);
end;

destructor  TPythonModule.Destroy;
begin
  FClients.Free;
  FErrors.Free;
  inherited;
end;

procedure TPythonModule.MakeModule;
begin
  CheckEngine;
  if Assigned(FModule) then
    Exit;
  with Engine do
    FModule := Py_InitModule( PChar(ModuleName), MethodsData );
end;

procedure TPythonModule.Initialize;
var
  i : Integer;
begin
  inherited;
  MakeModule;
  for i := 0 to ClientCount - 1 do
    if Clients[i] is TPythonType then
      with TPythonType(Clients[i]) do
        AddTypeVar;
  BuildErrors;
end;

procedure TPythonModule.AddClient( client : TEngineClient );
begin
  FClients.Add( client );
end;

function TPythonModule.ErrorByName( const name : String ) : TError;
var
  i : Integer;
begin
  for i := 0 to Errors.Count - 1 do
    if CompareText( Errors.Items[i].Name, name ) = 0 then
      begin
        Result := Errors.Items[i];
        Exit;
      end;
  raise Exception.CreateFmt( 'Could not find error "%s"', [name] );
end;

procedure TPythonModule.RaiseError( const error, msg : String );
begin
  ErrorByName( error ).RaiseError( msg );
end;

procedure TPythonModule.RaiseErrorFmt( const error, format : String; Args : array of const );
begin
  RaiseError( error, SysUtils.Format( format, Args ) );
end;

procedure TPythonModule.RaiseErrorObj( const error, msg : String; obj : PPyObject );
begin
  ErrorByName( error ).RaiseErrorObj( msg, obj );
end;

procedure TPythonModule.BuildErrors;
var
  i : Integer;
  d : PPyObject;
begin
  CheckEngine;
  with Engine do
    begin
      d := PyModule_GetDict( Module );
      if not Assigned(d) then
        Exit;
      for i := 0 to Errors.Count - 1 do
        with Errors.Items[i] do
          begin
            BuildError( ModuleName );
            PyDict_SetItemString( d, PChar(Name), Error );
          end;
    end;
end;

(*******************************************************)
(**                                                   **)
(**     class TPythonType                             **)
(**                                                   **)
(*******************************************************)

//////////////////////////////
//  TPyObject

// Constructors & Destructors
constructor TPyObject.Create( APythonType : TPythonType );
begin
  inherited Create;
  ob_refcnt := 1;
  PythonType := APythonType;
end;

constructor TPyObject.CreateWith( APythonType : TPythonType; args : PPyObject );
begin
  Create( APythonType );
end;

// Misc
function  TPyObject.GetSelf : PPyObject;
begin
  Result := PPyObject( @ob_refcnt );
end;

procedure TPyObject.Adjust(PyPointer: Pointer); assembler;
asm
  sub [PyPointer],ob_refcnt
end;

function  TPyObject.GetModule : TPythonModule;
begin
  if Assigned(PythonType) then
    Result := PythonType.Module
  else
    Result := nil;
end;


// Type services
////////////////

// Basic services
function  TPyObject.Print( var f: file; i: integer) : Integer;
begin
  Result := -1;
end;

function  TPyObject.GetAttr(key : PChar) : PPyObject;
begin
  with GetPythonEngine do
    begin
      // check for a method
      Result := Py_FindMethod( PythonType.MethodsData, GetSelf, key);
      if not Assigned(Result) then
        PyErr_SetString (PyExc_AttributeError^, PChar(Format('Unknown attribute "%s"',[key])));
    end;
end;

function  TPyObject.SetAttr(key : PChar; value : PPyObject) : Integer;
begin
  with GetPythonEngine do
    begin
      Result := -1;
      PyErr_SetString (PyExc_AttributeError^, PChar(Format('Unknown attribute "%s"',[key])));
    end;
end;

function  TPyObject.Repr : PPyObject;
begin
  with GetPythonEngine do
    Result := PyString_FromString( PChar(Format('<%s at %x>', [ClassName, Integer(self)])) );
end;

function  TPyObject.Compare( obj: PPyObject) : Integer;
begin
  Result := 0;
end;

function  TPyObject.Hash : Integer;
begin
  Result := Integer(Self);
end;

function  TPyObject.Str: PPyObject;
begin
  Result := Repr;
end;

function  TPyObject.GetAttrO( key: PPyObject) : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.SetAttrO( key, value: PPyObject) : Integer;
begin
  Result := -1;
end;

function  TPyObject.Call( ob1, ob2 : PPyObject) : PPyObject;
begin
  Result := nil;
end;

// Number services
function  TPyObject.NbAdd( obj : PPyObject) : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbSubstract( obj : PPyObject) : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbMultiply( obj : PPyObject) : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbDivide( obj : PPyObject) : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbRemainder( obj : PPyObject) : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbDivmod( obj : PPyObject) : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbPower( ob1, ob2 : PPyObject) : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbNegative : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbPositive : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbAbsolute : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbNonZero : Integer;
begin
  Result := -1;
end;

function  TPyObject.NbInvert : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbLShift( obj : PPyObject) : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbRShift( obj : PPyObject) : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbAnd( obj : PPyObject) : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbXor( obj : PPyObject) : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbOr( obj : PPyObject) : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbCoerce( obj : PPPyObject) : Integer;
begin
  Result := 0;
end;

function  TPyObject.NbInt : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbLong : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbFloat : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbOct : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.NbHex : PPyObject;
begin
  Result := nil;
end;

// Sequence services
function  TPyObject.SqLength : Integer;
begin
  Result := 0;
end;

function  TPyObject.SqConcat( obj : PPyObject) : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.SqRepeat( val : Integer ) : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.SqItem( idx : Integer ) : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.SqSlice( idx1, idx2 : Integer ) : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.SqAssItem( idx : integer; obj : PPyObject) : Integer;
begin
  Result := -1;
end;

function  TPyObject.SqAssSlice( idx1, idx2 : Integer; obj : PPyObject): integer;
begin
  Result := -1;
end;

// Mapping services
function  TPyObject.MpLength : Integer;
begin
  Result := 0;
end;

function  TPyObject.MpSubscript( obj : PPyObject) : PPyObject;
begin
  Result := nil;
end;

function  TPyObject.MpAssSubscript( obj1, obj2 : PPyObject) : Integer;
begin
  Result := -1;
end;


// Class methods
class procedure TPyObject.RegisterMethods( PythonType : TPythonType );
begin
end;


//////////////////////////////
//  TTypeServices

constructor TTypeServices.Create;
begin
  inherited;
  FBasic := [bsGetAttr, bsSetAttr, bsRepr, bsStr];
end;

procedure TTypeServices.AssignTo( Dest: TPersistent );
begin
  if Dest is TTypeServices then
     with TTypeServices( Dest ) do
       begin
         FBasic    := Self.FBasic;
         FNumber   := Self.FNumber;
         FSequence := Self.FSequence;
         FMapping  := Self.FMapping;
       end;
  inherited;
end;

//////////////////////////////
//  TPythonType

function  PythonToDelphi( obj : PPyObject ) : TPyObject;
begin
  Result := TPyObject(PChar(obj)-kObjectOffset);
end;

procedure PyObjectDestructor( pSelf : PPyObject); far; cdecl;
begin
  PythonToDelphi(pSelf).Free;
end;

procedure TPythonType.Notification( AComponent: TComponent;
                                    Operation: TOperation);
begin
  if Operation = opRemove then
    if AComponent = FModule then
      FModule := nil;
end;

procedure TPythonType.SetPyObjectClass( val : TPyObjectClass );
begin
  if val <> FPyObjectClass then
    begin
      FPyObjectClass := val;
      if Assigned(val) then
        begin
          FType.ob_size := val.InstanceSize - kObjectOffset;
          val.RegisterMethods( Self );
        end;
    end;
end;

procedure TPythonType.SetModule( val : TPythonModule );
begin
  if val <> FModule then
    begin
      FModule := val;
      if Assigned(val) then
        begin
          FCreateFuncName := FPrefix+FTypeName;
          val.AddDelphiMethod( PChar(FCreateFuncName), CreateMethod, '' );
        end;
    end;
end;

procedure TPythonType.SetServices( val : TTypeServices );
begin
  FServices.Assign( val );
end;

procedure TPythonType.SetTypeName( const val : String );
begin
  if (FTypeName <> val) and (val <> '') then
    begin
      FTypeName := val;
    end;
end;

function  TPythonType.CreateMethod( pSelf, args : PPyObject ) : PPyObject;
begin
  Result := CreateInstanceWith( args );
end;

procedure TPythonType.InitServices;
begin
  with FType do
    begin
      // Basic services
      tp_dealloc   := @PyObjectDestructor;
      if bsGetAttr in Services.Basic then
        tp_getattr   := GetCallBack( Self, @TPythonType.GetAttr, 2, cdecl );
      if bsSetAttr in Services.Basic then
        tp_setattr   := GetCallBack( Self, @TPythonType.SetAttr, 3, cdecl );
      if bsRepr in Services.Basic then
        tp_repr      := GetCallBack( Self, @TPythonType.Repr, 1, cdecl );
      if bsStr in Services.Basic then
        tp_str       := GetCallBack( Self, @TPythonType.Str, 1, cdecl );
      if bsCompare in Services.Basic then
        tp_compare   := GetCallBack( Self, @TPythonType.Compare, 2, cdecl );
      if bsHash in Services.Basic then
        tp_hash      := GetCallBack( Self, @TPythonType.Hash, 1, cdecl );
      if bsGetAttrO in Services.Basic then
        tp_getattro  := GetCallBack( Self, @TPythonType.GetAttrO, 2, cdecl );
      if bsSetAttrO in Services.Basic then
        tp_setattro  := GetCallBack( Self, @TPythonType.SetAttrO, 3, cdecl );
      if bsCall in Services.Basic then
        tp_call      := GetCallBack( Self, @TPythonType.Call, 3, cdecl );

      // Number services
      with tp_as_number do
        begin
          if nsAdd in Services.Number then
            nb_add := GetCallBack( Self, @TPythonType.NbAdd, 2, cdecl );
          if nsSubstract in Services.Number then
            nb_substract := GetCallBack( Self, @TPythonType.NbSubstract, 2, cdecl );
          if nsMultiply in Services.Number then
            nb_multiply := GetCallBack( Self, @TPythonType.NbMultiply, 2, cdecl );
          if nsDivide in Services.Number then
            nb_divide := GetCallBack( Self, @TPythonType.NbDivide, 2, cdecl );
          if nsRemainder in Services.Number then
            nb_remainder := GetCallBack( Self, @TPythonType.NbRemainder, 2, cdecl );
          if nsDivmod in Services.Number then
            nb_divmod := GetCallBack( Self, @TPythonType.NbDivmod, 2, cdecl );
          if nsPower in Services.Number then
            nb_power := GetCallBack( Self, @TPythonType.NbPower, 3, cdecl );
          if nsNegative in Services.Number then
            nb_negative := GetCallBack( Self, @TPythonType.NbNegative, 1, cdecl );
          if nsPositive in Services.Number then
            nb_positive := GetCallBack( Self, @TPythonType.NbPositive, 1, cdecl );
          if nsAbsolute in Services.Number then
            nb_absolute := GetCallBack( Self, @TPythonType.NbAbsolute, 1, cdecl );
          if nsNonZero in Services.Number then
            nb_nonzero := GetCallBack( Self, @TPythonType.NbNonZero, 1, cdecl );
          if nsInvert in Services.Number then
            nb_invert := GetCallBack( Self, @TPythonType.NbInvert, 1, cdecl );
          if nsLShift in Services.Number then
            nb_lshift := GetCallBack( Self, @TPythonType.NbLShift, 2, cdecl );
          if nsRShift in Services.Number then
            nb_rshift := GetCallBack( Self, @TPythonType.NbRShift, 2, cdecl );
          if nsAnd in Services.Number then
            nb_and := GetCallBack( Self, @TPythonType.NbAnd, 2, cdecl );
          if nsXor in Services.Number then
            nb_xor := GetCallBack( Self, @TPythonType.NbXor, 2, cdecl );
          if nsOr in Services.Number then
            nb_or := GetCallBack( Self, @TPythonType.NbOr, 2, cdecl );
          if nsCoerce in Services.Number then
            nb_coerce := GetCallBack( Self, @TPythonType.NbCoerce, 2, cdecl );
          if nsInt in Services.Number then
            nb_int := GetCallBack( Self, @TPythonType.NbInt, 1, cdecl );
          if nsLong in Services.Number then
            nb_long := GetCallBack( Self, @TPythonType.NbLong, 1, cdecl );
          if nsFloat in Services.Number then
            nb_float := GetCallBack( Self, @TPythonType.NbFloat, 1, cdecl );
          if nsOct in Services.Number then
            nb_oct := GetCallBack( Self, @TPythonType.NbOct, 1, cdecl );
          if nsHex in Services.Number then
            nb_hex := GetCallBack( Self, @TPythonType.NbHex, 1, cdecl );
        end;
      // Sequence services

      with tp_as_sequence do
        begin
          if ssLength in Services.Sequence then
            sq_length := GetCallBack( Self, @TPythonType.SqLength, 1, cdecl );
          if ssConcat in Services.Sequence then
            sq_concat := GetCallBack( Self, @TPythonType.SqConcat, 2, cdecl );
          if ssRepeat in Services.Sequence then
            sq_repeat := GetCallBack( Self, @TPythonType.SqRepeat, 2, cdecl );
          if ssItem in Services.Sequence then
            sq_item := GetCallBack( Self, @TPythonType.SqItem, 2, cdecl );
          if ssSlice in Services.Sequence then
            sq_slice := GetCallBack( Self, @TPythonType.SqSlice, 3, cdecl );
          if ssAssItem in Services.Sequence then
            sq_ass_item := GetCallBack( Self, @TPythonType.SqAssItem, 3, cdecl );
          if ssAssSlice in Services.Sequence then
            sq_ass_slice := GetCallBack( Self, @TPythonType.SqAssSlice, 4, cdecl );
        end;

      // Mapping services

      with tp_as_mapping do
        begin
          if msLength in Services.Mapping then
            mp_length := GetCallBack( Self, @TPythonType.MpLength, 1, cdecl );
          if msSubScript in Services.Mapping then
            mp_subscript := GetCallBack( Self, @TPythonType.MpSubscript, 2, cdecl );
          if msAssSubscript in Services.Mapping then
            mp_ass_subscript := GetCallBack( Self, @TPythonType.MpAssSubscript, 3, cdecl );
        end;
    end;
end;

// Type services
// They will be all forwarded to the Delphi class that
// implements the object through the use of virtual
// methods
///////////////////////////////////////

// Basic services

function  TPythonType.Print( pSelf : PPyObject; var f: file; i: integer) : Integer;
begin
  Result := PythonToDelphi(pSelf).Print( f, i );
end;

function  TPythonType.GetAttr( pSelf : PPyObject; key : PChar) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).GetAttr( key );
end;

function  TPythonType.SetAttr( pSelf : PPyObject; key : PChar; value : PPyObject) : Integer;
begin
  Result := PythonToDelphi(pSelf).SetAttr( key, value );
end;

function  TPythonType.Repr( pSelf : PPyObject ) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).Repr;
end;

function  TPythonType.Compare( pSelf, obj : PPyObject ) : Integer;
begin
  Result := PythonToDelphi(pSelf).Compare( obj );
end;

function  TPythonType.Hash( pSelf : PPyObject) : Integer;
begin
  Result := PythonToDelphi(pSelf).Hash;
end;

function  TPythonType.Str( pSelf : PPyObject ) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).Str;
end;

function  TPythonType.GetAttrO( pSelf, key: PPyObject) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).GetAttrO( key );
end;

function  TPythonType.SetAttrO( pSelf, key, value: PPyObject) : Integer;
begin
  Result := PythonToDelphi(pSelf).SetAttrO( key, value );
end;

function  TPythonType.Call( pSelf, ob1, ob2 : PPyObject) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).Call( ob1, ob2 );
end;

// Number services

function  TPythonType.NbAdd( pSelf, obj : PPyObject) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbAdd( obj );
end;

function  TPythonType.NbSubstract( pSelf, obj : PPyObject) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbSubstract( obj );
end;

function  TPythonType.NbMultiply( pSelf, obj : PPyObject) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbMultiply( obj );
end;

function  TPythonType.NbDivide( pSelf, obj : PPyObject) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbDivide( obj );
end;

function  TPythonType.NbRemainder( pSelf, obj : PPyObject) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbRemainder( obj );
end;

function  TPythonType.NbDivmod( pSelf, obj : PPyObject) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbDivmod( obj );
end;

function  TPythonType.NbPower( pSelf, ob1, ob2 : PPyObject) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbPower( ob1, ob2 );
end;

function  TPythonType.NbNegative( pSelf : PPyObject ) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbNegative;
end;

function  TPythonType.NbPositive( pSelf : PPyObject ) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbPositive;
end;

function  TPythonType.NbAbsolute( pSelf : PPyObject ) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbAbsolute;
end;

function  TPythonType.NbNonZero( pSelf : PPyObject ) : Integer;
begin
  Result := PythonToDelphi(pSelf).NbNonZero;
end;

function  TPythonType.NbInvert( pSelf : PPyObject ) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbInvert;
end;

function  TPythonType.NbLShift( pSelf, obj : PPyObject) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbLShift( obj );
end;

function  TPythonType.NbRShift( pSelf, obj : PPyObject) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbRShift( obj );
end;

function  TPythonType.NbAnd( pSelf, obj : PPyObject) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbAnd( obj );
end;

function  TPythonType.NbXor( pSelf, obj : PPyObject) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbXor( obj );
end;

function  TPythonType.NbOr( pSelf, obj : PPyObject) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbOr( obj );
end;

function  TPythonType.NbCoerce( pSelf : PPyobject; obj : PPPyObject) : Integer;
begin
  Result := PythonToDelphi(pSelf).NbCoerce( obj );
end;

function  TPythonType.NbInt( pSelf : PPyObject ) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbInt;
end;

function  TPythonType.NbLong( pSelf : PPyObject ) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbLong;
end;

function  TPythonType.NbFloat( pSelf : PPyObject ) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbFloat;
end;

function  TPythonType.NbOct( pSelf : PPyObject ) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbOct;
end;

function  TPythonType.NbHex( pSelf : PPyObject ) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).NbHex;
end;


// Sequence services

function  TPythonType.SqLength( pSelf : PPyObject ) : Integer;
begin
  Result := PythonToDelphi(pSelf).SqLength;
end;

function  TPythonType.SqConcat( pSelf, obj : PPyObject) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).SqConcat( obj );
end;

function  TPythonType.SqRepeat( pSelf : PPyObject; val : Integer ) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).SqRepeat( val );
end;

function  TPythonType.SqItem( pSelf : PPyObject; idx : Integer ) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).SqItem( idx );
end;

function  TPythonType.SqSlice( pSelf : PPyObject; idx1, idx2 : Integer ) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).SqSlice( idx1, idx2 );
end;

function  TPythonType.SqAssItem( pSelf : PPyObject; idx : integer; obj : PPyObject) : Integer;
begin
  Result := PythonToDelphi(pSelf).SqAssItem( idx, obj );
end;

function  TPythonType.SqAssSlice( pSelf : PPyObject; idx1, idx2 : Integer; obj : PPyObject): integer;
begin
  Result := PythonToDelphi(pSelf).SqAssSlice( idx1, idx2, obj );
end;


// Mapping services

function  TPythonType.MpLength( pSelf : PPyObject ) : Integer;
begin
  Result := PythonToDelphi(pSelf).MpLength;
end;

function  TPythonType.MpSubscript( pSelf, obj : PPyObject) : PPyObject;
begin
  Result := PythonToDelphi(pSelf).MpSubscript( obj );
end;

function  TPythonType.MpAssSubscript( pSelf, obj1, obj2 : PPyObject) : Integer;
begin
  Result := PythonToDelphi(pSelf).MpAssSubscript( obj1, obj2 );
end;

// Public methods

constructor TPythonType.Create( AOwner : TComponent );
begin
  inherited;
  FPrefix := 'Create';
  FServices := TTypeServices.Create;
end;

destructor  TPythonType.Destroy;
begin
  FServices.Free;
  inherited;
end;

function  TPythonType.GetTypePtr : PPyTypeObject;
begin
  Result := PPyTypeObject(@FType);
end;

procedure TPythonType.Initialize;
begin
  CheckEngine;
  with Engine, TheType do
    begin
      ob_type   := PPyTypeObject(PyType_Type);
      ob_refcnt := 1;
      tp_name   := PChar(FTypeName);
    end;
  if Assigned(FModule) then
    begin
      if Module.Initialized then
        AddTypeVar
      else
        Module.AddClient( Self );
    end;
  InitServices;
  inherited;
end;

function TPythonType.CreateInstance : PPyObject;
var
  obj : TPyObject;
begin
  with GetPythonEngine do
    begin
      obj := PyObjectClass.Create( Self );
      obj.ob_type := @FType;
      if PyErr_Occurred <> nil then
        Result := nil
      else
        Result := obj.GetSelf;
    end;
end;

function TPythonType.CreateInstanceWith( args : PPyObject ) : PPyObject;
var
  obj : TPyObject;
begin
  with GetPythonEngine do
    begin
      obj := PyObjectClass.CreateWith( Self, args );
      obj.ob_type := @FType;
      if PyErr_Occurred <> nil then
        Result := nil
      else
        Result := obj.GetSelf;
    end;
end;

procedure TPythonType.AddTypeVar;
var
  d : PPyObject;
begin
  CheckEngine;
  with Engine do
    begin
      d := PyModule_GetDict( Module.Module );
      PyDict_SetItemString( d, PChar(TypeName), PPyObject(TheTypePtr) );
    end;
end;

(*******************************************************)
(**                                                   **)
(**     class TPythonDelphiVar                        **)
(**                                                   **)
(*******************************************************)

procedure TPythonDelphiVar.CreateVarType;
begin
  FVarType := TPythonType.Create( Owner );
  with FVarType do
    begin
      TypeName := 'PythonDelphiVar';
      Engine := Self.Engine;
      with TheType do
        begin
          tp_basicsize := sizeof(PythonDelphiVar);
          tp_dealloc   := PyVar_dealloc;
          tp_getattr   := PyVar_getattr;
          tp_setattr   := PyVar_setattrfunc;
          tp_repr      := PyVar_repr;
          tp_str       := PyVar_repr;
        end;
    end;
end;

procedure TPythonDelphiVar.CreateVar;
var
  v : PPythonDelphiVar;
  m, d : PPyObject;
begin
  if not Assigned(Engine) then
    Exit;
  with Engine do
    begin
      // Create an instance of PythonDelphiVar
      new(v);
      with v^ do
        begin
          ob_refcnt := 1;
          ob_type := FVarType.TheTypePtr;
          dv_component := Self;
        end;
      FVarObject := PPyObject(v);
      // Add a reference to this var in the module
      m := PyImport_AddModule(PChar(Module));
      if m = nil then
        raise EPythonError.CreateFmt('CreateVar: can''t create module "%s"', [Module]);
      d := PyModule_GetDict(m);
      PyDict_SetItemString( d, PChar(VarName), FVarObject );
    end;
end;

function  TPythonDelphiVar.GetValue : Variant;
begin
  if Assigned( FVarObject ) then
    with PPythonDelphiVar(FVarObject)^ do
      Result := dv_var
  else
    raise Exception.Create('No variable was created' );
end;

procedure TPythonDelphiVar.SetValue( val : Variant );
begin
  if Assigned( FVarObject ) then
    with PPythonDelphiVar(FVarObject)^ do
      dv_var := val
  else
    raise Exception.Create('No variable was created' );
end;

function  TPythonDelphiVar.GetValueAsString : String;
begin
  Result := Value;
end;

procedure TPythonDelphiVar.SetVarName( const val : String );

  procedure CheckVarName;
  var
    i : Integer;
  begin
    if Owner = nil then Exit;
    if (val = FVarName) or (val = '') then Exit;
    for i := 0 to Owner.ComponentCount - 1 do
      if Owner.Components[i] is TPythonDelphiVar then
        with TPythonDelphiVar(Owner.Components[i]) do
          if (VarName = val) and (Module = Self.Module) then
            raise Exception.CreateFmt('A variable "%s" already exists in the module "%s"',[val, Module]);
  end;

begin
  if val <> FVarName then
    begin
      CheckVarName;
      FVarName := val;
    end;
end;

constructor TPythonDelphiVar.Create( AOwner : TComponent );

  procedure AdjustName;
  var
    i, cpt : Integer;
    done : Boolean;
  begin
    if AOwner = nil then Exit;
    cpt := 1;
    done := False;
    while not done do
      begin
        done := True;
        for i := 0 to AOwner.ComponentCount - 1 do
          if AOwner.Components[i] is TPythonDelphiVar then
            with TPythonDelphiVar(AOwner.Components[i]) do
              if (VarName = Self.FVarName+IntToStr(cpt)) and
                 (Module = Self.Module) then
                begin
                  Inc(cpt);
                  done := False;
                  Break;
                end;
      end;
    FVarName := FVarName + IntToStr(cpt);
  end;

begin
  inherited;
  FModule := '__main__';
  FVarName := 'varname';
  AdjustName;
end;

procedure TPythonDelphiVar.Initialize;
begin
  inherited;
  if csDesigning in ComponentState then
    Exit;
  CreateVarType;
  CreateVar;
end;

procedure TPythonDelphiVar.Finalize;
begin
  inherited;
  if not Assigned(Engine) then
    Exit;
  if Assigned(FVarObject) then
    with PPythonDelphiVar(FVarObject)^ do
      dv_component := nil;
  with Engine do
    Py_XDECREF( FVarObject );
end;



(*******************************************************)
(**                                                   **)
(**     Methods for new Python objects or modules     **)
(**                                                   **)
(*******************************************************)

/////////////////////////////////////////////////////////
// Module pyio for Python Input/Outputs
//

function pyio_write(self, args : PPyObject) : PPyObject;
var
  a1 : PPyObject;
begin
  with GetPythonEngine do
    begin
      if Assigned(args) and (PyTuple_Size(args) > 0) then
        begin
          a1 := PyTuple_GetItem(args, 0);
          if RedirectIO and (IO <> nil) and
             Assigned(a1) and PyString_Check(a1) then
            IO.Write(PyObjectAsString(a1));
          Result := ReturnNone;
        end
      else
        begin
          PyErr_BadArgument;
          Result := nil;
        end;
    end;
end;

function pyio_read(self, args : PPyObject) : PPyObject;
var
  txt, msg : String;
begin
  with GetPythonEngine do
    begin
      if RedirectIO  then
        begin
          txt := '';
          msg := 'Enter text';
          if Assigned(IO) then
            txt := IO.ReceiveData;
          Result := PyString_FromString(PChar(txt));
        end
      else
        Result := ReturnNone;
    end;
end;

function pyio_SetDelayWrites(self, args : PPyObject) : PPyObject;
var
  val : Integer;
begin
  with GetPythonEngine do
    begin
      if PyArg_ParseTuple( args, 'i:SetDelayWrites', [@val] ) <> 0 then
        begin
          if IO <> nil then
            IO.DelayWrites := val <> 0;
          Result := ReturnNone;
        end
      else
        Result := nil;
    end;
end;

function pyio_SetMaxLines(self, args : PPyObject) : PPyObject;
var
  val : Integer;
begin
  with GetPythonEngine do
    begin
      if PyArg_ParseTuple( args, 'i:SetMaxLines', [@val] ) <> 0 then
        begin
          if IO <> nil then
            IO .MaxLines := val;
          Result := ReturnNone;
        end
      else
        Result := nil;
    end;
end;

/////////////////////////////////////////////////////////
// Functions for the new Type Object PythonDelphiVar
//

function PyVar_GetVar(obj : PPyObject) : Variant;
begin
  with GetPythonEngine, PPythonDelphiVar(obj)^ do
    begin
      if Assigned( dv_component ) and
         (@dv_component.OnGetData <> nil) then
        dv_component.OnGetData( dv_component, dv_var );
      Result := dv_var;
    end;
end;

function PyVar_AsObject( obj : PPyObject; V : Variant ) : PPyObject;
begin
  with GetPythonEngine, PPythonDelphiVar(obj)^ do
    Result := VariantAsPyObject( V );
end;

function PyVar_FromObject( obj, value : PPyObject ) : Integer;
begin
  Result := 0;
  with GetPythonEngine, PPythonDelphiVar(obj)^ do
    begin
      dv_var := PyObjectAsVariant( value );
      if Assigned( dv_component ) and
         (@dv_component.OnSetData <> nil) then
        dv_component.OnSetData( dv_component, dv_var );
      if Assigned( dv_component ) and
         (@dv_component.OnChange <> nil) then
        dv_component.OnChange( dv_component );
    end;
end;

procedure PyVar_dealloc(obj : PPyObject); cdecl;
begin
  Dispose( obj );
end;

function  PyVar_getattr(obj : PPyObject; key : PChar) : PPyObject; cdecl;
begin
  with GetPythonEngine, PPythonDelphiVar(obj)^ do
    begin
      // Check for attribute Value
      if UpperCase(key) = 'VALUE' then
        Result := PyVar_AsObject( obj, PyVar_GetVar(obj) )
      else
        begin
          Result := nil;
          PyErr_SetString (PyExc_AttributeError^, PChar(Format('Unknown attribute "%s"',[key])));
        end;
    end;
end;

function  PyVar_setattrfunc(obj : PPyObject; key : PChar; value : PPyObject) : Integer; cdecl;
begin
  Result := -1;
  with GetPythonEngine, PPythonDelphiVar(obj)^ do
    begin
      // Check for attribute value
      if UpperCase(key) = 'VALUE' then begin
        Result := PyVar_FromObject( obj, value );
      end else
        PyErr_SetString (PyExc_AttributeError^, PChar(Format('Unknown attribute "%s"',[key])));
    end;
end;

function  PyVar_repr(obj : PPyObject) : PPyObject; cdecl;
var
  tmp : Variant;
begin
  with GetPythonEngine, PPythonDelphiVar(obj)^ do
    begin
      tmp := PyVar_GetVar(obj);
      if VarType(tmp) in [varEmpty, varNull] then
        Result := PyObject_Repr( Py_None )
      else
        Result := PyString_FromString( PChar(Format('%s',[tmp]) ) );
    end;
end;

(*******************************************************)
(**                                                   **)
(**            Global procedures                      **)
(**                                                   **)
(*******************************************************)

function  GetPythonEngine : TPythonEngine;
begin
  if not Assigned( gPythonEngine ) then
    raise Exception.Create( 'No Python engine was created' );
  if not gPythonEngine.Initialized then
    raise Exception.Create( 'The Python engine is not properly initialized' );
  Result := gPythonEngine;
end;

procedure Register;
begin
  RegisterComponents('Python',[ TPythonInterface, TPythonEngine, TPythonInputOutput,
                                TPythonType, TPythonModule, TPythonDelphiVar]);
end;

end.
